/*
 * Decompiled with CFR 0.152.
 */
package org.geowebcache.s3;

import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.iterable.S3Objects;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Predicate;
import java.util.logging.Level;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import javax.annotation.Nullable;
import org.apache.commons.io.IOUtils;
import org.geowebcache.GeoWebCacheException;
import org.geowebcache.locks.LockProvider;
import org.geowebcache.locks.NoOpLockProvider;
import org.geowebcache.s3.S3BlobStore;
import org.geowebcache.storage.StorageException;
import org.geowebcache.util.TMSKeyBuilder;

class S3Ops {
    private final AmazonS3Client conn;
    private final String bucketName;
    private final TMSKeyBuilder keyBuilder;
    private final LockProvider locks;
    private ExecutorService deleteExecutorService;
    private Map<String, Long> pendingDeletesKeyTime = new ConcurrentHashMap<String, Long>();

    public S3Ops(AmazonS3Client conn, String bucketName, TMSKeyBuilder keyBuilder, LockProvider locks) throws StorageException {
        this.conn = conn;
        this.bucketName = bucketName;
        this.keyBuilder = keyBuilder;
        this.locks = locks == null ? new NoOpLockProvider() : locks;
        this.deleteExecutorService = this.createDeleteExecutorService();
        this.issuePendingBulkDeletes();
    }

    private ExecutorService createDeleteExecutorService() {
        ThreadFactory tf = new ThreadFactoryBuilder().setDaemon(true).setNameFormat("GWC S3BlobStore bulk delete thread-%d. Bucket: " + this.bucketName).setPriority(1).build();
        return Executors.newCachedThreadPool(tf);
    }

    public void shutDown() {
        this.deleteExecutorService.shutdownNow();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void issuePendingBulkDeletes() throws StorageException {
        LockProvider.Lock lock;
        String pendingDeletesKey = this.keyBuilder.pendingDeletes();
        try {
            lock = this.locks.getLock(pendingDeletesKey);
        }
        catch (GeoWebCacheException e) {
            throw new StorageException("Unable to lock pending deletes", (Throwable)e);
        }
        try {
            Properties deletes = this.getProperties(pendingDeletesKey);
            for (Map.Entry<Object, Object> e : deletes.entrySet()) {
                String prefix = e.getKey().toString();
                long timestamp = Long.parseLong(e.getValue().toString());
                S3BlobStore.log.info(String.format("Restarting pending bulk delete on '%s/%s':%d", this.bucketName, prefix, timestamp));
                this.asyncDelete(prefix, timestamp);
            }
        }
        finally {
            try {
                lock.release();
            }
            catch (GeoWebCacheException e) {
                throw new StorageException("Unable to unlock pending deletes", (Throwable)e);
            }
        }
    }

    private void clearPendingBulkDelete(String prefix, long timestamp) throws GeoWebCacheException {
        Long taskTime = this.pendingDeletesKeyTime.get(prefix);
        if (taskTime == null) {
            return;
        }
        if (taskTime > timestamp) {
            return;
        }
        String pendingDeletesKey = this.keyBuilder.pendingDeletes();
        LockProvider.Lock lock = this.locks.getLock(pendingDeletesKey);
        try {
            long storedTimestamp;
            Properties deletes = this.getProperties(pendingDeletesKey);
            String storedVal = (String)deletes.remove(prefix);
            long l = storedTimestamp = storedVal == null ? Long.MIN_VALUE : Long.parseLong(storedVal);
            if (timestamp >= storedTimestamp) {
                this.putProperties(pendingDeletesKey, deletes);
            } else {
                S3BlobStore.log.info(String.format("bulk delete finished but there's a newer one ongoing for bucket '%s/%s'", this.bucketName, prefix));
            }
        }
        catch (StorageException e) {
            throw new RuntimeException(e);
        }
        finally {
            lock.release();
        }
    }

    public boolean scheduleAsyncDelete(String prefix) throws GeoWebCacheException {
        long timestamp = this.currentTimeSeconds();
        String msg = String.format("Issuing bulk delete on '%s/%s' for objects older than %d", this.bucketName, prefix, timestamp);
        S3BlobStore.log.info(msg);
        LockProvider.Lock lock = this.locks.getLock(prefix);
        try {
            boolean taskRuns = this.asyncDelete(prefix, timestamp);
            if (taskRuns) {
                String pendingDeletesKey = this.keyBuilder.pendingDeletes();
                Properties deletes = this.getProperties(pendingDeletesKey);
                deletes.setProperty(prefix, String.valueOf(timestamp));
                this.putProperties(pendingDeletesKey, deletes);
            }
            boolean bl = taskRuns;
            return bl;
        }
        catch (StorageException e) {
            throw new RuntimeException(e);
        }
        finally {
            lock.release();
        }
    }

    private long currentTimeSeconds() {
        long timestamp = (long)Math.ceil((double)System.currentTimeMillis() / 1000.0) * 1000L;
        return timestamp;
    }

    private synchronized boolean asyncDelete(String prefix, long timestamp) {
        if (!this.prefixExists(prefix)) {
            return false;
        }
        Long currentTaskTime = this.pendingDeletesKeyTime.get(prefix);
        if (currentTaskTime != null && currentTaskTime > timestamp) {
            return false;
        }
        BulkDelete task = new BulkDelete((AmazonS3)this.conn, this.bucketName, prefix, timestamp);
        this.deleteExecutorService.submit(task);
        this.pendingDeletesKeyTime.put(prefix, timestamp);
        return true;
    }

    @Nullable
    public ObjectMetadata getObjectMetadata(String key) throws StorageException {
        ObjectMetadata obj;
        block2: {
            obj = null;
            try {
                obj = this.conn.getObjectMetadata(this.bucketName, key);
            }
            catch (AmazonS3Exception e) {
                if (404 == e.getStatusCode()) break block2;
                throw new StorageException("Error checking existence of " + key + ": " + e.getMessage(), (Throwable)e);
            }
        }
        return obj;
    }

    public void putObject(PutObjectRequest putObjectRequest) throws StorageException {
        try {
            this.conn.putObject(putObjectRequest);
        }
        catch (RuntimeException e) {
            throw new StorageException("Error storing " + putObjectRequest.getKey(), (Throwable)e);
        }
    }

    @Nullable
    public S3Object getObject(String key) throws StorageException {
        S3Object object;
        try {
            object = this.conn.getObject(this.bucketName, key);
        }
        catch (AmazonS3Exception e) {
            if (404 == e.getStatusCode()) {
                return null;
            }
            throw new StorageException("Error fetching " + key + ": " + e.getMessage(), (Throwable)e);
        }
        if (this.isPendingDelete(object)) {
            this.closeObject(object);
            return null;
        }
        return object;
    }

    private void closeObject(S3Object object) throws StorageException {
        try {
            object.close();
        }
        catch (IOException e) {
            throw new StorageException("Error closing connection to " + object.getKey() + ": " + e.getMessage(), (Throwable)e);
        }
    }

    public boolean deleteObject(String key) {
        try {
            this.conn.deleteObject(this.bucketName, key);
        }
        catch (AmazonS3Exception e) {
            return false;
        }
        return true;
    }

    private boolean isPendingDelete(S3Object object) {
        if (this.pendingDeletesKeyTime.isEmpty()) {
            return false;
        }
        String key = object.getKey();
        long lastModified = object.getObjectMetadata().getLastModified().getTime();
        for (Map.Entry<String, Long> e : this.pendingDeletesKeyTime.entrySet()) {
            String parentKey = e.getKey();
            if (!key.startsWith(parentKey)) continue;
            long deleteTime = e.getValue();
            return deleteTime >= lastModified;
        }
        return false;
    }

    /*
     * Enabled aggressive exception aggregation
     */
    @Nullable
    public byte[] getBytes(String key) throws StorageException {
        try (S3Object object = this.getObject(key);){
            byte[] byArray;
            block16: {
                if (object == null) {
                    byte[] byArray2 = null;
                    return byArray2;
                }
                S3ObjectInputStream in = object.getObjectContent();
                try {
                    byte[] bytes;
                    byArray = bytes = IOUtils.toByteArray((InputStream)in);
                    if (in == null) break block16;
                }
                catch (Throwable throwable) {
                    if (in != null) {
                        try {
                            in.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                in.close();
            }
            return byArray;
        }
        catch (IOException e) {
            throw new StorageException("Error getting " + key, (Throwable)e);
        }
    }

    public boolean prefixExists(String prefix) {
        boolean hasNext = S3Objects.withPrefix((AmazonS3)this.conn, (String)this.bucketName, (String)prefix).withBatchSize(1).iterator().hasNext();
        return hasNext;
    }

    public Properties getProperties(String key) {
        byte[] bytes;
        Properties properties = new Properties();
        try {
            bytes = this.getBytes(key);
        }
        catch (StorageException e) {
            throw new RuntimeException(e);
        }
        if (bytes != null) {
            try {
                properties.load(new InputStreamReader((InputStream)new ByteArrayInputStream(bytes), StandardCharsets.UTF_8));
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
        return properties;
    }

    public void putProperties(String resourceKey, Properties properties) throws StorageException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        try {
            properties.store(out, "");
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        byte[] bytes = out.toByteArray();
        ObjectMetadata objectMetadata = new ObjectMetadata();
        objectMetadata.setContentLength((long)bytes.length);
        objectMetadata.setContentType("text/plain");
        ByteArrayInputStream in = new ByteArrayInputStream(bytes);
        PutObjectRequest putReq = new PutObjectRequest(this.bucketName, resourceKey, (InputStream)in, objectMetadata);
        this.putObject(putReq);
    }

    public Stream<S3ObjectSummary> objectStream(String prefix) {
        return StreamSupport.stream(S3Objects.withPrefix((AmazonS3)this.conn, (String)this.bucketName, (String)prefix).spliterator(), false);
    }

    private static class TimeStampFilter
    implements Predicate<S3ObjectSummary> {
        private long timeStamp;

        public TimeStampFilter(long timeStamp) {
            this.timeStamp = timeStamp;
        }

        @Override
        public boolean test(S3ObjectSummary summary) {
            long lastModified = summary.getLastModified().getTime();
            boolean applies = this.timeStamp >= lastModified;
            return applies;
        }
    }

    private class BulkDelete
    implements Callable<Long> {
        private final String prefix;
        private final long timestamp;
        private final AmazonS3 conn;
        private final String bucketName;

        public BulkDelete(AmazonS3 conn, String bucketName, String prefix, long timestamp) {
            this.conn = conn;
            this.bucketName = bucketName;
            this.prefix = prefix;
            this.timestamp = timestamp;
        }

        @Override
        public Long call() throws Exception {
            long count = 0L;
            try {
                this.checkInterrupted();
                S3BlobStore.log.info(String.format("Running bulk delete on '%s/%s':%d", this.bucketName, this.prefix, this.timestamp));
                TimeStampFilter filter = new TimeStampFilter(this.timestamp);
                AtomicInteger n = new AtomicInteger(0);
                Collection<List<S3ObjectSummary>> partitions = S3Ops.this.objectStream(this.prefix).filter(filter).collect(Collectors.groupingBy(x -> n.getAndIncrement() % 1000)).values();
                for (List list : partitions) {
                    this.checkInterrupted();
                    ArrayList<DeleteObjectsRequest.KeyVersion> keys = new ArrayList<DeleteObjectsRequest.KeyVersion>(list.size());
                    for (S3ObjectSummary so : list) {
                        String key = so.getKey();
                        keys.add(new DeleteObjectsRequest.KeyVersion(key));
                    }
                    this.checkInterrupted();
                    if (keys.isEmpty()) continue;
                    DeleteObjectsRequest deleteReq = new DeleteObjectsRequest(this.bucketName);
                    deleteReq.setQuiet(true);
                    deleteReq.setKeys(keys);
                    this.checkInterrupted();
                    this.conn.deleteObjects(deleteReq);
                    count += (long)keys.size();
                }
            }
            catch (IllegalStateException | InterruptedException e) {
                S3BlobStore.log.info(String.format("S3 bulk delete aborted for '%s/%s'. Will resume on next startup.", this.bucketName, this.prefix));
                throw e;
            }
            catch (Exception e) {
                S3BlobStore.log.log(Level.WARNING, String.format("Unknown error performing bulk S3 delete of '%s/%s'", this.bucketName, this.prefix), e);
                throw e;
            }
            S3BlobStore.log.info(String.format("Finished bulk delete on '%s/%s':%d. %d objects deleted", this.bucketName, this.prefix, this.timestamp, count));
            S3Ops.this.clearPendingBulkDelete(this.prefix, this.timestamp);
            return count;
        }

        private void checkInterrupted() throws InterruptedException {
            if (Thread.interrupted()) {
                throw new InterruptedException();
            }
        }
    }
}

