/*
 * Decompiled with CFR 0.152.
 */
package org.geowebcache.s3;

import com.amazonaws.AmazonServiceException;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.model.AccessControlList;
import com.amazonaws.services.s3.model.BucketPolicy;
import com.amazonaws.services.s3.model.CannedAccessControlList;
import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.AbstractIterator;
import com.google.common.collect.Iterators;
import com.google.common.collect.Lists;
import com.google.common.collect.UnmodifiableIterator;
import com.google.common.io.ByteStreams;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.channels.Channels;
import java.nio.channels.WritableByteChannel;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.geotools.util.logging.Logging;
import org.geowebcache.GeoWebCacheException;
import org.geowebcache.filter.parameters.ParametersUtils;
import org.geowebcache.io.ByteArrayResource;
import org.geowebcache.io.Resource;
import org.geowebcache.layer.TileLayerDispatcher;
import org.geowebcache.locks.LockProvider;
import org.geowebcache.mime.MimeException;
import org.geowebcache.mime.MimeType;
import org.geowebcache.s3.S3BlobStoreInfo;
import org.geowebcache.s3.S3Ops;
import org.geowebcache.storage.BlobStore;
import org.geowebcache.storage.BlobStoreListener;
import org.geowebcache.storage.BlobStoreListenerList;
import org.geowebcache.storage.CompositeBlobStore;
import org.geowebcache.storage.StorageException;
import org.geowebcache.storage.TileObject;
import org.geowebcache.storage.TileRange;
import org.geowebcache.storage.TileRangeIterator;
import org.geowebcache.util.TMSKeyBuilder;

public class S3BlobStore
implements BlobStore {
    static Logger log = Logging.getLogger((String)S3BlobStore.class.getName());
    private final BlobStoreListenerList listeners = new BlobStoreListenerList();
    private AmazonS3Client conn;
    private final TMSKeyBuilder keyBuilder;
    private String bucketName;
    private volatile boolean shutDown;
    private final S3Ops s3Ops;
    private CannedAccessControlList acl;

    public S3BlobStore(S3BlobStoreInfo config, TileLayerDispatcher layers, LockProvider lockProvider) throws StorageException {
        Preconditions.checkNotNull((Object)((Object)config));
        Preconditions.checkNotNull((Object)layers);
        this.bucketName = config.getBucket();
        String prefix = config.getPrefix() == null ? "" : config.getPrefix();
        this.keyBuilder = new TMSKeyBuilder(prefix, layers);
        this.conn = this.validateClient(config.buildClient(), this.bucketName);
        this.acl = config.getAccessControlList();
        this.s3Ops = new S3Ops(this.conn, this.bucketName, this.keyBuilder, lockProvider);
        boolean empty = !this.s3Ops.prefixExists(prefix);
        boolean existing = Objects.nonNull(this.s3Ops.getObjectMetadata(this.keyBuilder.storeMetadata()));
        CompositeBlobStore.checkSuitability((String)config.getLocation(), (boolean)existing, (boolean)empty);
        this.s3Ops.putProperties(this.keyBuilder.storeMetadata(), new Properties());
    }

    protected AmazonS3Client validateClient(AmazonS3Client client, String bucketName) throws StorageException {
        List<S3ClientChecker> connectionCheckers = Arrays.asList(this::checkBucketPolicy, this::checkAccessControlList);
        ArrayList<Exception> exceptions = new ArrayList<Exception>();
        for (S3ClientChecker checker : connectionCheckers) {
            try {
                checker.validate(client, bucketName);
                break;
            }
            catch (Exception e2) {
                exceptions.add(e2);
            }
        }
        if (exceptions.size() == connectionCheckers.size()) {
            String messages = exceptions.stream().map(e -> e.getMessage()).collect(Collectors.joining("\n"));
            throw new StorageException("Could not validate the connection to S3, exceptions gathered during checks:\n " + messages);
        }
        return client;
    }

    private void checkAccessControlList(AmazonS3Client client, String bucketName) throws Exception {
        try {
            log.fine("Checking access rights to bucket " + bucketName);
            AccessControlList bucketAcl = client.getBucketAcl(bucketName);
            List grants = bucketAcl.getGrantsAsList();
            log.fine("Bucket " + bucketName + " permissions: " + String.valueOf(grants));
        }
        catch (AmazonServiceException se) {
            throw new StorageException("Server error listing bucket ACLs: " + se.getMessage(), (Throwable)se);
        }
    }

    private void checkBucketPolicy(AmazonS3Client client, String bucketName) throws Exception {
        try {
            log.fine("Checking policy for bucket " + bucketName);
            BucketPolicy bucketPol = client.getBucketPolicy(bucketName);
            log.fine("Bucket " + bucketName + " policy: " + bucketPol.getPolicyText());
        }
        catch (AmazonServiceException se) {
            throw new StorageException("Server error getting bucket policy: " + se.getMessage(), (Throwable)se);
        }
    }

    public void destroy() {
        this.shutDown = true;
        AmazonS3Client conn = this.conn;
        this.conn = null;
        if (conn != null) {
            this.s3Ops.shutDown();
            conn.shutdown();
        }
    }

    public void addListener(BlobStoreListener listener) {
        this.listeners.addListener(listener);
    }

    public boolean removeListener(BlobStoreListener listener) {
        return this.listeners.removeListener(listener);
    }

    public void put(TileObject obj) throws StorageException {
        ObjectMetadata oldObj;
        boolean existed;
        String mimeType;
        Resource blob = obj.getBlob();
        Preconditions.checkNotNull((Object)blob);
        Preconditions.checkNotNull((Object)obj.getBlobFormat());
        String key = this.keyBuilder.forTile(obj);
        ObjectMetadata objectMetadata = new ObjectMetadata();
        objectMetadata.setContentLength(blob.getSize());
        String blobFormat = obj.getBlobFormat();
        try {
            mimeType = MimeType.createFromFormat((String)blobFormat).getMimeType();
        }
        catch (MimeException me) {
            throw new RuntimeException(me);
        }
        objectMetadata.setContentType(mimeType);
        if (this.listeners.isEmpty()) {
            existed = false;
            oldObj = null;
        } else {
            oldObj = this.s3Ops.getObjectMetadata(key);
            existed = oldObj != null;
        }
        ByteArrayInputStream input = this.toByteArray(blob);
        PutObjectRequest putObjectRequest = new PutObjectRequest(this.bucketName, key, (InputStream)input, objectMetadata).withCannedAcl(this.acl);
        log.finer((String)(log.isLoggable(Level.FINER) ? "Storing " + key : ""));
        this.s3Ops.putObject(putObjectRequest);
        this.putParametersMetadata(obj.getLayerName(), obj.getParametersId(), obj.getParameters());
        if (!this.listeners.isEmpty()) {
            if (existed) {
                long oldSize = oldObj.getContentLength();
                this.listeners.sendTileUpdated(obj, oldSize);
            } else {
                this.listeners.sendTileStored(obj);
            }
        }
    }

    private ByteArrayInputStream toByteArray(Resource blob) throws StorageException {
        byte[] bytes;
        if (blob instanceof ByteArrayResource) {
            bytes = ((ByteArrayResource)blob).getContents();
        } else {
            try (ByteArrayOutputStream out = new ByteArrayOutputStream((int)blob.getSize());
                 WritableByteChannel channel = Channels.newChannel(out);){
                blob.transferTo(channel);
                bytes = out.toByteArray();
            }
            catch (IOException e) {
                throw new StorageException("Error copying blob contents", (Throwable)e);
            }
        }
        ByteArrayInputStream input = new ByteArrayInputStream(bytes);
        return input;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public boolean get(TileObject obj) throws StorageException {
        String key = this.keyBuilder.forTile(obj);
        try (S3Object object = this.s3Ops.getObject(key);){
            if (object == null) {
                boolean bl = false;
                return bl;
            }
            try (S3ObjectInputStream in = object.getObjectContent();){
                byte[] bytes = ByteStreams.toByteArray((InputStream)in);
                obj.setBlobSize(bytes.length);
                obj.setBlob((Resource)new ByteArrayResource(bytes));
                obj.setCreated(object.getObjectMetadata().getLastModified().getTime());
                return true;
            }
        }
        catch (IOException e) {
            throw new StorageException("Error getting " + key, (Throwable)e);
        }
    }

    public boolean delete(final TileRange tileRange) throws StorageException {
        String coordsPrefix = this.keyBuilder.coordinatesPrefix(tileRange, true);
        if (!this.s3Ops.prefixExists(coordsPrefix)) {
            return false;
        }
        AbstractIterator<long[]> tileLocations = new AbstractIterator<long[]>(){
            private TileRangeIterator trIter;
            {
                this.trIter = new TileRangeIterator(tileRange, new int[]{1, 1});
            }

            protected long[] computeNext() {
                long[] gridLoc = this.trIter.nextMetaGridLocation(new long[3]);
                return gridLoc == null ? (long[])this.endOfData() : gridLoc;
            }
        };
        if (this.listeners.isEmpty()) {
            UnmodifiableIterator partition = Iterators.partition((Iterator)tileLocations, (int)1000);
            TileToKey tileToKey = new TileToKey(coordsPrefix, tileRange.getMimeType());
            while (partition.hasNext() && !this.shutDown) {
                List locations = (List)partition.next();
                List keys = Lists.transform((List)locations, (Function)tileToKey);
                DeleteObjectsRequest req = new DeleteObjectsRequest(this.bucketName);
                req.setQuiet(true);
                req.setKeys(keys);
                this.conn.deleteObjects(req);
            }
        } else {
            String layerName = tileRange.getLayerName();
            String gridSetId = tileRange.getGridSetId();
            String format = tileRange.getMimeType().getFormat();
            Map parameters = tileRange.getParameters();
            while (tileLocations.hasNext()) {
                long[] xyz = (long[])tileLocations.next();
                TileObject tile = TileObject.createQueryTileObject((String)layerName, (long[])xyz, (String)gridSetId, (String)format, (Map)parameters);
                tile.setParametersId(tileRange.getParametersId());
                this.delete(tile);
            }
        }
        return true;
    }

    public boolean delete(String layerName) throws StorageException {
        boolean layerExists;
        Preconditions.checkNotNull((Object)layerName, (Object)"layerName");
        String metadataKey = this.keyBuilder.layerMetadata(layerName);
        String layerPrefix = this.keyBuilder.forLayer(layerName);
        this.s3Ops.deleteObject(metadataKey);
        try {
            layerExists = this.s3Ops.scheduleAsyncDelete(layerPrefix);
        }
        catch (GeoWebCacheException e) {
            throw new RuntimeException(e);
        }
        if (layerExists) {
            this.listeners.sendLayerDeleted(layerName);
        }
        return layerExists;
    }

    public boolean deleteByGridsetId(String layerName, String gridSetId) throws StorageException {
        boolean prefixExists;
        Preconditions.checkNotNull((Object)layerName, (Object)"layerName");
        Preconditions.checkNotNull((Object)gridSetId, (Object)"gridSetId");
        String gridsetPrefix = this.keyBuilder.forGridset(layerName, gridSetId);
        try {
            prefixExists = this.s3Ops.scheduleAsyncDelete(gridsetPrefix);
        }
        catch (GeoWebCacheException e) {
            throw new RuntimeException(e);
        }
        if (prefixExists) {
            this.listeners.sendGridSubsetDeleted(layerName, gridSetId);
        }
        return prefixExists;
    }

    public boolean delete(TileObject obj) throws StorageException {
        String key = this.keyBuilder.forTile(obj);
        if (this.listeners.isEmpty()) {
            return this.s3Ops.deleteObject(key);
        }
        ObjectMetadata oldObj = this.s3Ops.getObjectMetadata(key);
        if (oldObj == null) {
            return false;
        }
        this.s3Ops.deleteObject(key);
        obj.setBlobSize((int)oldObj.getContentLength());
        this.listeners.sendTileDeleted(obj);
        return true;
    }

    public boolean rename(String oldLayerName, String newLayerName) throws StorageException {
        log.fine("No need to rename layers, S3BlobStore uses layer id as key root");
        if (this.s3Ops.prefixExists(oldLayerName)) {
            this.listeners.sendLayerRenamed(oldLayerName, newLayerName);
        }
        return true;
    }

    public void clear() throws StorageException {
        throw new UnsupportedOperationException("clear() should not be called");
    }

    @Nullable
    public String getLayerMetadata(String layerName, String key) {
        Properties properties = this.getLayerMetadata(layerName);
        String value = properties.getProperty(key);
        return value;
    }

    public void putLayerMetadata(String layerName, String key, String value) {
        Properties properties = this.getLayerMetadata(layerName);
        properties.setProperty(key, value);
        String resourceKey = this.keyBuilder.layerMetadata(layerName);
        try {
            this.s3Ops.putProperties(resourceKey, properties);
        }
        catch (StorageException e) {
            throw new RuntimeException(e);
        }
    }

    private Properties getLayerMetadata(String layerName) {
        String key = this.keyBuilder.layerMetadata(layerName);
        return this.s3Ops.getProperties(key);
    }

    private void putParametersMetadata(String layerName, String parametersId, Map<String, String> parameters) {
        assert (Objects.isNull(parametersId) == Objects.isNull(parameters));
        if (Objects.isNull(parametersId)) {
            return;
        }
        Properties properties = new Properties();
        parameters.forEach(properties::setProperty);
        String resourceKey = this.keyBuilder.parametersMetadata(layerName, parametersId);
        try {
            this.s3Ops.putProperties(resourceKey, properties);
        }
        catch (StorageException e) {
            throw new RuntimeException(e);
        }
    }

    public boolean layerExists(String layerName) {
        String coordsPrefix = this.keyBuilder.forLayer(layerName);
        boolean layerExists = this.s3Ops.prefixExists(coordsPrefix);
        return layerExists;
    }

    public boolean deleteByParametersId(String layerName, String parametersId) throws StorageException {
        Preconditions.checkNotNull((Object)layerName, (Object)"layerName");
        Preconditions.checkNotNull((Object)parametersId, (Object)"parametersId");
        boolean prefixExists = this.keyBuilder.forParameters(layerName, parametersId).stream().map(prefix -> {
            try {
                return this.s3Ops.scheduleAsyncDelete((String)prefix);
            }
            catch (RuntimeException | GeoWebCacheException e) {
                throw new RuntimeException(e);
            }
        }).reduce(Boolean::logicalOr).orElse(false);
        if (prefixExists) {
            this.listeners.sendParametersDeleted(layerName, parametersId);
        }
        return prefixExists;
    }

    public Set<Map<String, String>> getParameters(String layerName) {
        return this.s3Ops.objectStream(this.keyBuilder.parametersMetadataPrefix(layerName)).map(S3ObjectSummary::getKey).map(this.s3Ops::getProperties).map(props -> props).collect(Collectors.toSet());
    }

    public Map<String, Optional<Map<String, String>>> getParametersMapping(String layerName) {
        return this.s3Ops.objectStream(this.keyBuilder.parametersMetadataPrefix(layerName)).map(S3ObjectSummary::getKey).map(this.s3Ops::getProperties).map(props -> props).collect(Collectors.toMap(ParametersUtils::getId, Optional::of));
    }

    private class TileToKey
    implements Function<long[], DeleteObjectsRequest.KeyVersion> {
        private final String coordsPrefix;
        private final String extension;

        public TileToKey(String coordsPrefix, MimeType mimeType) {
            this.coordsPrefix = coordsPrefix;
            this.extension = mimeType.getInternalName();
        }

        public DeleteObjectsRequest.KeyVersion apply(long[] loc) {
            long z = loc[2];
            long x = loc[0];
            long y = loc[1];
            StringBuilder sb = new StringBuilder(this.coordsPrefix);
            sb.append(z).append('/').append(x).append('/').append(y).append('.').append(this.extension);
            return new DeleteObjectsRequest.KeyVersion(sb.toString());
        }
    }

    static interface S3ClientChecker {
        public void validate(AmazonS3Client var1, String var2) throws Exception;
    }
}

