/*
 *    GeoTools - The Open Source Java GIS Toolkit
 *    http://geotools.org
 *
 *    (C) 2002-2008, Open Source Geospatial Foundation (OSGeo)
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation;
 *    version 2.1 of the License.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 */
package org.geotools.xsd;

import static org.junit.Assert.assertTrue;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.StringReader;
import java.net.MalformedURLException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.xml.namespace.QName;
import junit.framework.TestCase;
import org.geotools.ml.MLConfiguration;
import org.geotools.ml.Mail;
import org.geotools.ml.bindings.MLSchemaLocationResolver;
import org.geotools.xsd.impl.Handler;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.ext.EntityResolver2;

public class ParserTest extends TestCase {

    public void testParseEntityResolver() throws Exception {
        Parser parser = new Parser(new XMLConfiguration());
        AtomicBoolean resolverUsed = new AtomicBoolean(false);
        parser.setEntityResolver(
                new EntityResolver2() {
                    @Override
                    public InputSource getExternalSubset(String name, String baseURI)
                            throws SAXException, IOException {
                        return null;
                    }

                    @Override
                    public InputSource resolveEntity(
                            String name, String publicId, String baseURI, String systemId)
                            throws SAXException, IOException {
                        if (systemId.equals("./mails.xsd")) resolverUsed.set(true);
                        return null;
                    }

                    @Override
                    public InputSource resolveEntity(String publicId, String systemId)
                            throws SAXException, IOException {
                        if (systemId.equals("./mails.xsd")) resolverUsed.set(true);
                        return null;
                    }
                });
        parser.parse(MLSchemaLocationResolver.class.getResourceAsStream("mails-local-schema.xml"));
        assertTrue("The resolver was not used?", resolverUsed.get());
    }

    public void testParse() throws Exception {
        Parser parser = new Parser(new MLConfiguration());
        List mails =
                (List)
                        parser.parse(
                                MLSchemaLocationResolver.class.getResourceAsStream("mails.xml"));

        assertEquals(2, mails.size());

        Mail mail = (Mail) mails.get(0);
        assertEquals(0, mail.getId().intValue());

        mail = (Mail) mails.get(1);
        assertEquals(1, mail.getId().intValue());
    }

    public void testParseValid() throws Exception {
        Parser parser = new Parser(new MLConfiguration());
        parser.setValidating(true);
        parser.parse(MLSchemaLocationResolver.class.getResourceAsStream("mails.xml"));

        assertEquals(0, parser.getValidationErrors().size());
    }

    public void testParseNull() throws Exception {
        Parser parser = new Parser(new MLConfiguration());
        parser.setValidating(true);
        List mails =
                (List)
                        parser.parse(
                                MLSchemaLocationResolver.class.getResourceAsStream(
                                        "null-mail.xml"));

        assertEquals(0, parser.getValidationErrors().size());
        assertEquals(1, mails.size());

        Mail mail = (Mail) mails.get(0);
        assertNull(mail.getBody());
    }

    public void testParseInValid() throws Exception {
        Parser parser = new Parser(new MLConfiguration());
        parser.setValidating(true);
        parser.parse(MLSchemaLocationResolver.class.getResourceAsStream("mails-invalid.xml"));

        assertFalse(0 == parser.getValidationErrors().size());

        // test immeediate failure case
        parser.setFailOnValidationError(true);
        try {
            parser.parse(MLSchemaLocationResolver.class.getResourceAsStream("mails-invalid.xml"));
            fail("should have thrown an error with setFailOnValidationError set");
        } catch (SAXException e) {
        }
    }

    public void testValidate() throws Exception {
        Parser parser = new Parser(new MLConfiguration());
        parser.validate(MLSchemaLocationResolver.class.getResourceAsStream("mails-invalid.xml"));

        assertFalse(0 == parser.getValidationErrors().size());

        // test immeediate failure case
        parser.setFailOnValidationError(true);
        try {
            parser.validate(
                    MLSchemaLocationResolver.class.getResourceAsStream("mails-invalid.xml"));
            fail("should have thrown an error with setFailOnValidationError set");
        } catch (SAXException e) {
        }
    }

    public void testParserDelegate() throws Exception {
        MLConfiguration config = new MLConfiguration();

        MyParserDelegate delegate = new MyParserDelegate();
        assertFalse(delegate.foo);
        assertFalse(delegate.bar);

        config.getContext().registerComponentInstance(delegate);

        Parser parser = new Parser(config);
        Object o = parser.parse(ParserTest.class.getResourceAsStream("parserDelegate.xml"));

        assertTrue(delegate.foo);
        assertTrue(delegate.bar);
    }

    static class MyParserDelegate implements ParserDelegate, ParserDelegate2 {

        boolean foo = false;
        boolean bar = false;

        public boolean canHandle(QName elementName) {
            return "parserDelegateElement".equals(elementName.getLocalPart());
        }

        public void initialize(QName elementName) {}

        public Object getParsedObject() {
            return null;
        }

        public void characters(char[] ch, int start, int length) throws SAXException {
            if (!bar && "bar".equals(new String(ch, start, length))) {
                bar = true;
            }
        }

        public void endDocument() throws SAXException {}

        public void endElement(String uri, String localName, String name) throws SAXException {}

        public void endPrefixMapping(String prefix) throws SAXException {}

        public void ignorableWhitespace(char[] ch, int start, int length) throws SAXException {}

        public void processingInstruction(String target, String data) throws SAXException {}

        public void setDocumentLocator(Locator locator) {}

        public void skippedEntity(String name) throws SAXException {}

        public void startDocument() throws SAXException {}

        public void startElement(String uri, String localName, String name, Attributes atts)
                throws SAXException {
            if (!foo && "foo".equals(localName)) {
                foo = true;
            }
        }

        public void startPrefixMapping(String prefix, String uri) throws SAXException {}

        @Override
        public boolean canHandle(
                QName elementName, Attributes attributes, Handler handler, Handler parent) {
            return canHandle(elementName);
        }
    }

    public void testMixedContent() throws Exception {
        final StringBuffer sb = new StringBuffer();
        XSD xsd =
                new XSD() {
                    @Override
                    public String getSchemaLocation() {
                        return ParserTest.class.getResource("mixed.xsd").getFile();
                    }

                    @Override
                    public String getNamespaceURI() {
                        return "http://geotools.org/test";
                    }
                };
        Configuration cfg =
                new Configuration(xsd) {
                    @Override
                    protected void registerBindings(Map bindings) {
                        bindings.put(
                                new QName("http://geotools.org/test", "MixedType"),
                                new MixedTypeBinding(sb));
                    }

                    @Override
                    protected void configureParser(Parser parser) {
                        parser.setHandleMixedContent(true);
                    }
                };

        Parser p = new Parser(cfg);

        p.parse(getClass().getResourceAsStream("mixed1.xml"));
        assertEquals("Hello 'there' how are 'you'?", sb.toString());

        sb.setLength(0);
        p.parse(getClass().getResourceAsStream("mixed2.xml"));
        assertEquals("Hello 'there' how are 'you' ?", sb.toString());
    }

    public static class MixedTypeBinding extends AbstractComplexBinding {
        StringBuffer sb = new StringBuffer();

        public MixedTypeBinding(StringBuffer sb) {
            this.sb = sb;
        }

        public QName getTarget() {
            return new QName("http://geotools.org/test", "MixedType");
        }

        public Class getType() {
            return Object.class;
        }

        @Override
        public Object parse(ElementInstance instance, Node node, Object value) throws Exception {
            for (Node n : ((List<Node>) node.getChildren())) {
                if (n.getValue() instanceof Text) {
                    sb.append(((Text) n.getValue()).getValue());
                } else {
                    sb.append("'").append(n.getValue()).append("'");
                }
            }
            return value;
        }
    }

    /**
     * Test Parser with an XML document containing an external entity: <!ENTITY c SYSTEM
     * "file:///this/file/does/not/exist">
     */
    public void testParseWithEntityResolver() throws Exception {
        Parser parser = new Parser(new MLConfiguration());

        try {
            parser.parse(
                    MLSchemaLocationResolver.class.getResourceAsStream(
                            "mails-external-entities.xml"));
            fail("parsing should throw an exception since referenced file does not exist");
        } catch (FileNotFoundException e) {
        }
        try {
            parser.validate(
                    MLSchemaLocationResolver.class.getResourceAsStream(
                            "mails-external-entities.xml"));
            fail("validating should throw an exception since referenced file does not exist");
        } catch (FileNotFoundException e) {
        }

        // Set an EntityResolver implementation to prevent usage of external entities.
        // When parsing an XML entity, the empty InputSource returned by this resolver provokes
        // a java.net.MalformedURLException
        parser.setEntityResolver(
                new EntityResolver2() {
                    @Override
                    public InputSource resolveEntity(String publicId, String systemId)
                            throws SAXException, IOException {
                        return new InputSource();
                    }

                    @Override
                    public InputSource getExternalSubset(String name, String baseURI)
                            throws SAXException, IOException {
                        return new InputSource();
                    }

                    @Override
                    public InputSource resolveEntity(
                            String name, String publicId, String baseURI, String systemId)
                            throws SAXException, IOException {
                        return new InputSource();
                    }
                });

        try {
            parser.parse(
                    MLSchemaLocationResolver.class.getResourceAsStream(
                            "mails-external-entities.xml"));
            fail("parsing an XML with external entities should throw a MalformedURLException");
        } catch (MalformedURLException e) {
        }
        try {
            parser.validate(
                    MLSchemaLocationResolver.class.getResourceAsStream(
                            "mails-external-entities.xml"));
            fail("validating an XML with external entities should throw a MalformedURLException");
        } catch (MalformedURLException e) {
        }

        // Set another EntityResolver
        parser.setEntityResolver(
                new EntityResolver2() {
                    @Override
                    public InputSource resolveEntity(String publicId, String systemId)
                            throws SAXException, IOException {
                        if ("file:///this/file/does/not/exist".equals(systemId)) {
                            return new InputSource(new StringReader("hello"));
                        } else {
                            return new InputSource();
                        }
                    }

                    @Override
                    public InputSource getExternalSubset(String name, String baseURI)
                            throws SAXException, IOException {
                        // TODO Auto-generated method stub
                        return null;
                    }

                    @Override
                    public InputSource resolveEntity(
                            String name, String publicId, String baseURI, String systemId)
                            throws SAXException, IOException {
                        if ("file:///this/file/does/not/exist".equals(systemId)) {
                            return new InputSource(new StringReader("hello"));
                        } else {
                            return new InputSource();
                        }
                    }
                });

        // parsing shouldn't throw an exception
        parser.parse(
                MLSchemaLocationResolver.class.getResourceAsStream("mails-external-entities.xml"));
        parser.validate(
                MLSchemaLocationResolver.class.getResourceAsStream("mails-external-entities.xml"));
    }

    /** Tests returned exception caused by entity expansion limit configuration on Parser. */
    public void testEntityExpansionLimitException() throws Exception {
        final StringBuffer sb = new StringBuffer();
        XSD xsd =
                new XSD() {
                    @Override
                    public String getSchemaLocation() {
                        return ParserTest.class.getResource("mixed.xsd").getFile();
                    }

                    @Override
                    public String getNamespaceURI() {
                        return "http://geotools.org/test";
                    }
                };
        Configuration cfg =
                new Configuration(xsd) {
                    @Override
                    protected void registerBindings(Map bindings) {
                        bindings.put(
                                new QName("http://geotools.org/test", "MixedType"),
                                new MixedTypeBinding(sb));
                    }

                    @Override
                    protected void configureParser(Parser parser) {
                        parser.setHandleMixedContent(true);
                    }
                };

        Parser p = new Parser(cfg);
        p.setEntityExpansionLimit(1);
        SAXParseException expected = null;
        try {
            p.parse(getClass().getResourceAsStream("entityExpansionLimit.xml"));
        } catch (SAXParseException ex) {
            expected = ex;
        }
        assertNotNull(expected);
        // check for the entity expansion limit error code in exception message
        assertTrue(expected.getMessage().contains("JAXP00010001"));
    }

    /** Tests entity expansion limit configuration on Parser. */
    public void testEntityExpansionLimitAllowed() throws Exception {
        final StringBuffer sb = new StringBuffer();
        XSD xsd =
                new XSD() {
                    @Override
                    public String getSchemaLocation() {
                        return ParserTest.class.getResource("mixed.xsd").getFile();
                    }

                    @Override
                    public String getNamespaceURI() {
                        return "http://geotools.org/test";
                    }
                };
        Configuration cfg =
                new Configuration(xsd) {
                    @Override
                    protected void registerBindings(Map bindings) {
                        bindings.put(
                                new QName("http://geotools.org/test", "MixedType"),
                                new MixedTypeBinding(sb));
                    }

                    @Override
                    protected void configureParser(Parser parser) {
                        parser.setHandleMixedContent(true);
                    }
                };

        Parser p = new Parser(cfg);
        p.setEntityExpansionLimit(100);
        SAXParseException unexpected = null;
        try {
            p.parse(getClass().getResourceAsStream("entityExpansionLimit.xml"));
        } catch (SAXParseException ex) {
            unexpected = ex;
        }
        assertNull(unexpected);
    }
}
