/*
 * Decompiled with CFR 0.152.
 */
package org.geotools.data.hana.wkb;

import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.text.MessageFormat;
import org.geotools.data.hana.wkb.GeometryType;
import org.geotools.data.hana.wkb.HanaWKBParserException;
import org.geotools.data.hana.wkb.XyzmMode;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.CoordinateSequence;
import org.locationtech.jts.geom.CoordinateSequenceFactory;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryCollection;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.LinearRing;
import org.locationtech.jts.geom.MultiLineString;
import org.locationtech.jts.geom.MultiPoint;
import org.locationtech.jts.geom.MultiPolygon;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;

public class HanaWKBParser {
    private static final byte XDR = 0;
    private static final byte NDR = 1;
    private static final int TYPE_MASK = 1048575;
    private static final int XYZM_MODE_DIV = 1000;
    private static final int XYZM_MODE_XY = 0;
    private static final int XYZM_MODE_XYZ = 1;
    private static final int XYZM_MODE_XYM = 2;
    private static final int XYZM_MODE_XYZM = 3;
    private GeometryFactory factory;
    private ByteBuffer data;
    private XyzmMode xyzmMode;
    private int dimension;
    private int jtsDimension;

    public HanaWKBParser(GeometryFactory factory) {
        this.factory = factory;
    }

    public Geometry parse(byte[] wkb) throws HanaWKBParserException {
        this.data = ByteBuffer.wrap(wkb);
        try {
            this.readAndSetByteOrder();
            int typeCode = this.data.getInt();
            GeometryType type = this.getGeometryType(typeCode);
            this.xyzmMode = this.getXyzmMode(typeCode);
            this.dimension = this.xyzmMode.getCoordinatesPerPoint();
            this.jtsDimension = this.xyzmMode.hasM() ? this.dimension - 1 : this.dimension;
            Geometry geometry = this.parseGeometryOfType(type);
            if (this.data.hasRemaining()) {
                throw new HanaWKBParserException("There is unparsed WKB data left");
            }
            return geometry;
        }
        catch (BufferUnderflowException e) {
            throw new HanaWKBParserException("WKB is too short", e);
        }
    }

    private Geometry parseGeometryOfType(GeometryType type) throws HanaWKBParserException {
        switch (type) {
            case POINT: {
                return this.parsePoint();
            }
            case LINESTRING: {
                return this.parseLineString();
            }
            case POLYGON: {
                return this.parsePolygon();
            }
            case MULTIPOINT: {
                return this.parseMultiPoint();
            }
            case MULTILINESTRING: {
                return this.parseMultiLineString();
            }
            case MULTIPOLYGON: {
                return this.parseMultiPolygon();
            }
            case GEOMETRYCOLLECTION: {
                return this.parseGeometryCollection();
            }
            case CIRCULARSTRING: {
                throw new HanaWKBParserException("Circular strings are not supported by JTS");
            }
        }
        throw new AssertionError();
    }

    private Point parsePoint() {
        Coordinate coord;
        double x = this.data.getDouble();
        double y = this.data.getDouble();
        if (this.xyzmMode.hasZ()) {
            double z = this.data.getDouble();
            coord = new Coordinate(x, y, z);
        } else {
            coord = new Coordinate(x, y);
        }
        if (this.xyzmMode.hasM()) {
            this.data.getDouble();
        }
        return this.factory.createPoint(coord);
    }

    private LineString parseLineString() {
        CoordinateSequence cs = this.readCoordinateSequence();
        return this.factory.createLineString(cs);
    }

    private Polygon parsePolygon() {
        int numRings = this.data.getInt();
        if (numRings == 0) {
            return this.factory.createPolygon((LinearRing)null);
        }
        LinearRing shell = this.parseLinearRing();
        if (numRings == 1) {
            return this.factory.createPolygon(shell);
        }
        LinearRing[] holes = new LinearRing[numRings - 1];
        for (int i = 1; i < numRings; ++i) {
            holes[i - 1] = this.parseLinearRing();
        }
        return this.factory.createPolygon(shell, holes);
    }

    private MultiPoint parseMultiPoint() throws HanaWKBParserException {
        int numPoints = this.data.getInt();
        Point[] points = new Point[numPoints];
        for (int i = 0; i < numPoints; ++i) {
            points[i] = (Point)this.parseSubGeometry(GeometryType.MULTIPOINT);
        }
        return this.factory.createMultiPoint(points);
    }

    private MultiLineString parseMultiLineString() throws HanaWKBParserException {
        int numLineStrings = this.data.getInt();
        LineString[] lineStrings = new LineString[numLineStrings];
        for (int i = 0; i < numLineStrings; ++i) {
            lineStrings[i] = (LineString)this.parseSubGeometry(GeometryType.MULTILINESTRING);
        }
        return this.factory.createMultiLineString(lineStrings);
    }

    private MultiPolygon parseMultiPolygon() throws HanaWKBParserException {
        int numPolygons = this.data.getInt();
        Polygon[] polygons = new Polygon[numPolygons];
        for (int i = 0; i < numPolygons; ++i) {
            polygons[i] = (Polygon)this.parseSubGeometry(GeometryType.MULTIPOLYGON);
        }
        return this.factory.createMultiPolygon(polygons);
    }

    private GeometryCollection parseGeometryCollection() throws HanaWKBParserException {
        int numGeometries = this.data.getInt();
        Geometry[] geometries = new Geometry[numGeometries];
        for (int i = 0; i < numGeometries; ++i) {
            geometries[i] = this.parseSubGeometry(GeometryType.GEOMETRYCOLLECTION);
        }
        return this.factory.createGeometryCollection(geometries);
    }

    private Geometry parseSubGeometry(GeometryType parentType) throws HanaWKBParserException {
        this.readAndSetByteOrder();
        int typeCode = this.data.getInt();
        GeometryType type = this.getGeometryType(typeCode);
        return this.parseGeometryOfType(type);
    }

    private LinearRing parseLinearRing() {
        CoordinateSequence cs = this.patchRing(this.readCoordinateSequence());
        return this.factory.createLinearRing(cs);
    }

    private CoordinateSequence patchRing(CoordinateSequence cs) {
        int i;
        if (cs.size() >= 4 || cs.size() == 0) {
            return cs;
        }
        Coordinate[] coords = new Coordinate[4];
        for (i = 0; i < cs.size(); ++i) {
            coords[i] = cs.getCoordinate(i);
        }
        for (i = cs.size(); i < 4; ++i) {
            coords[i] = cs.getCoordinate(0);
        }
        CoordinateSequenceFactory csf = this.factory.getCoordinateSequenceFactory();
        return csf.create(coords);
    }

    private CoordinateSequence readCoordinateSequence() {
        CoordinateSequenceFactory csf = this.factory.getCoordinateSequenceFactory();
        int numPoints = this.data.getInt();
        CoordinateSequence cs = csf.create(numPoints, this.jtsDimension);
        switch (this.xyzmMode) {
            case XY: {
                for (int i = 0; i < numPoints; ++i) {
                    cs.setOrdinate(i, 0, this.data.getDouble());
                    cs.setOrdinate(i, 1, this.data.getDouble());
                }
                break;
            }
            case XYZ: {
                for (int i = 0; i < numPoints; ++i) {
                    cs.setOrdinate(i, 0, this.data.getDouble());
                    cs.setOrdinate(i, 1, this.data.getDouble());
                    cs.setOrdinate(i, 2, this.data.getDouble());
                }
                break;
            }
            case XYM: {
                for (int i = 0; i < numPoints; ++i) {
                    cs.setOrdinate(i, 0, this.data.getDouble());
                    cs.setOrdinate(i, 1, this.data.getDouble());
                    this.data.getDouble();
                }
                break;
            }
            case XYZM: {
                for (int i = 0; i < numPoints; ++i) {
                    cs.setOrdinate(i, 0, this.data.getDouble());
                    cs.setOrdinate(i, 1, this.data.getDouble());
                    cs.setOrdinate(i, 2, this.data.getDouble());
                    this.data.getDouble();
                }
                break;
            }
        }
        return cs;
    }

    private GeometryType getGeometryType(int typeCode) throws HanaWKBParserException {
        int wkbType = typeCode & 0xFFFFF;
        GeometryType type = GeometryType.getFromCode(wkbType %= 1000);
        if (type == null) {
            throw new HanaWKBParserException(MessageFormat.format("Unknown WKB type {0}", wkbType));
        }
        return type;
    }

    private XyzmMode getXyzmMode(int typeCode) throws HanaWKBParserException {
        int wkbType = typeCode & 0xFFFFF;
        int xyzmFlag = wkbType / 1000;
        switch (xyzmFlag) {
            case 0: {
                return XyzmMode.XY;
            }
            case 1: {
                return XyzmMode.XYZ;
            }
            case 2: {
                return XyzmMode.XYM;
            }
            case 3: {
                return XyzmMode.XYZM;
            }
        }
        throw new HanaWKBParserException(MessageFormat.format("Invalid XYZM-mode {0}", xyzmFlag));
    }

    private void readAndSetByteOrder() throws HanaWKBParserException {
        byte order = this.data.get();
        switch (order) {
            case 0: {
                this.data.order(ByteOrder.BIG_ENDIAN);
                break;
            }
            case 1: {
                this.data.order(ByteOrder.LITTLE_ENDIAN);
                break;
            }
            default: {
                throw new HanaWKBParserException(MessageFormat.format("Invalid BOM value {0}", order));
            }
        }
    }
}

