/*
 * Decompiled with CFR 0.152.
 */
package org.geotools.tpk;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.image.BufferedImage;
import java.awt.image.Raster;
import java.awt.image.RenderedImage;
import java.awt.image.SampleModel;
import java.awt.image.WritableRaster;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import javax.imageio.ImageIO;
import javax.imageio.ImageReadParam;
import javax.imageio.ImageReader;
import javax.imageio.stream.ImageInputStream;
import org.geotools.api.coverage.grid.Format;
import org.geotools.api.geometry.Bounds;
import org.geotools.api.parameter.GeneralParameterValue;
import org.geotools.api.parameter.ParameterValue;
import org.geotools.api.referencing.ReferenceIdentifier;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;
import org.geotools.coverage.CoverageFactoryFinder;
import org.geotools.coverage.grid.GridCoverage2D;
import org.geotools.coverage.grid.GridEnvelope2D;
import org.geotools.coverage.grid.GridGeometry2D;
import org.geotools.coverage.grid.io.AbstractGridCoverage2DReader;
import org.geotools.coverage.grid.io.AbstractGridFormat;
import org.geotools.geometry.GeneralBounds;
import org.geotools.geometry.jts.ReferencedEnvelope;
import org.geotools.referencing.CRS;
import org.geotools.tpk.TPKFile;
import org.geotools.tpk.TPKFormat;
import org.geotools.tpk.TPKTile;
import org.geotools.tpk.TPKZoomLevel;
import org.geotools.util.factory.Hints;
import org.geotools.util.logging.Logging;

public class TPKReader
extends AbstractGridCoverage2DReader {
    private static final Logger LOGGER = Logging.getLogger(TPKReader.class);
    static final CoordinateReferenceSystem SPHERICAL_MERCATOR;
    static final CoordinateReferenceSystem WGS_84;
    protected static final ReferencedEnvelope WORLD_ENVELOPE;
    protected static final int DEFAULT_TILE_SIZE = 256;
    protected static final int ZOOM_LEVEL_BASE = 2;
    protected ReferencedEnvelope bounds;
    protected String imageFormat;
    protected File sourceFile;
    protected Map<Long, TPKZoomLevel> zoomLevelMap;

    public TPKReader(Object source, Hints hints) {
        long startConstructor = System.currentTimeMillis();
        this.sourceFile = TPKFormat.getFileFromSource(source);
        this.zoomLevelMap = new HashMap<Long, TPKZoomLevel>();
        TPKFile file = new TPKFile(this.sourceFile, this.zoomLevelMap);
        try {
            this.bounds = ReferencedEnvelope.create((Bounds)file.getBounds(), (CoordinateReferenceSystem)WGS_84).transform(SPHERICAL_MERCATOR, true);
        }
        catch (Exception e) {
            this.bounds = null;
        }
        this.originalEnvelope = new GeneralBounds((Bounds)(this.bounds == null ? WORLD_ENVELOPE : this.bounds));
        this.imageFormat = file.getImageFormat();
        long maxZoom = file.getMaxZoomLevel();
        long size = Math.round(Math.pow(2.0, maxZoom)) * 256L;
        this.highestRes = new double[]{WORLD_ENVELOPE.getSpan(0) / (double)size, WORLD_ENVELOPE.getSpan(1) / (double)size};
        this.originalGridRange = new GridEnvelope2D(new Rectangle((int)size, (int)size));
        this.coverageFactory = CoverageFactoryFinder.getGridCoverageFactory((Hints)this.hints);
        this.crs = SPHERICAL_MERCATOR;
        file.close();
        String msg = String.format("TPKReader constructor finished in %d milliseconds", System.currentTimeMillis() - startConstructor);
        LOGGER.fine(msg);
    }

    public Format getFormat() {
        return new TPKFormat();
    }

    public GridCoverage2D read(GeneralParameterValue[] parameters) throws IllegalArgumentException {
        long startRead = System.currentTimeMillis();
        TPKFile file = new TPKFile(this.sourceFile, this.zoomLevelMap, (Bounds)this.bounds, this.imageFormat);
        ReferencedEnvelope requestedEnvelope = null;
        Rectangle dim = null;
        if (parameters != null) {
            for (GeneralParameterValue parameter : parameters) {
                ParameterValue param = (ParameterValue)parameter;
                ReferenceIdentifier name = param.getDescriptor().getName();
                if (!name.equals(AbstractGridFormat.READ_GRIDGEOMETRY2D.getName())) continue;
                GridGeometry2D gg = (GridGeometry2D)param.getValue();
                try {
                    requestedEnvelope = ReferencedEnvelope.create((Bounds)gg.getEnvelope(), (CoordinateReferenceSystem)gg.getCoordinateReferenceSystem()).transform(SPHERICAL_MERCATOR, true);
                }
                catch (Exception e) {
                    requestedEnvelope = null;
                }
                dim = gg.getGridRange2D().getBounds();
            }
        }
        if (requestedEnvelope == null) {
            requestedEnvelope = this.bounds;
        }
        long zoomLevel = 0L;
        if (requestedEnvelope != null && dim != null) {
            double ratioWidth = requestedEnvelope.getSpan(0) / WORLD_ENVELOPE.getSpan(0);
            double propWidth = dim.getWidth() / ratioWidth;
            zoomLevel = Math.round(Math.log(propWidth / 256.0) / Math.log(2.0));
        }
        zoomLevel = file.getClosestZoom(zoomLevel);
        long numberOfTiles = Math.round(Math.pow(2.0, zoomLevel));
        double resX = WORLD_ENVELOPE.getSpan(0) / (double)numberOfTiles;
        double resY = WORLD_ENVELOPE.getSpan(1) / (double)numberOfTiles;
        double offsetX = WORLD_ENVELOPE.getMinimum(0);
        double offsetY = WORLD_ENVELOPE.getMinimum(1);
        long leftTile = file.getMinColumn(zoomLevel);
        long rightTile = file.getMaxColumn(zoomLevel);
        long bottomTile = file.getMinRow(zoomLevel);
        long topTile = file.getMaxRow(zoomLevel);
        if (requestedEnvelope != null) {
            leftTile = this.boundMax(leftTile, (requestedEnvelope.getMinimum(0) - offsetX) / resX);
            bottomTile = this.boundMax(bottomTile, (requestedEnvelope.getMinimum(1) - offsetY) / resY);
            rightTile = this.boundMinMax(leftTile, rightTile, (requestedEnvelope.getMaximum(0) - offsetX) / resX);
            topTile = this.boundMinMax(bottomTile, topTile, (requestedEnvelope.getMaximum(1) - offsetY) / resY);
        }
        int width = (int)(rightTile - leftTile + 1L) * 256;
        int height = (int)(topTile - bottomTile + 1L) * 256;
        ReferencedEnvelope resultEnvelope = new ReferencedEnvelope(offsetX + (double)leftTile * resX, offsetX + (double)(rightTile + 1L) * resX, offsetY + (double)bottomTile * resY, offsetY + (double)(topTile + 1L) * resY, SPHERICAL_MERCATOR);
        String imageFormat = file.getImageFormat();
        List<TPKTile> tiles = file.getTiles(zoomLevel, topTile, bottomTile, leftTile, rightTile, imageFormat);
        BufferedImage image = this.getStartImage(2, width, height);
        Graphics graphics = image.getGraphics();
        long originLeft = leftTile;
        long originTop = topTile;
        tiles.parallelStream().map(TileImage::new).forEach(tileImage -> {
            if (tileImage.image != null) {
                int posx = (int)(tileImage.col - originLeft) * 256;
                int posy = (int)(originTop - tileImage.row) * 256;
                graphics.drawImage(tileImage.image, posx, posy, null);
            }
        });
        file.close();
        String msg = String.format("At zoom level %d TPK read completed in %d milliseconds", zoomLevel, System.currentTimeMillis() - startRead);
        LOGGER.fine(msg);
        return this.coverageFactory.create((CharSequence)"unnamed", (RenderedImage)image, (Bounds)resultEnvelope);
    }

    private long boundMinMax(long max, long min, double value) {
        return Math.max(max, Math.min(min, Math.round(Math.floor(value))));
    }

    private long boundMax(long bound, double value) {
        return Math.max(bound, Math.round(Math.floor(value)));
    }

    private static String getImageFormat(byte[] imageData, String format) {
        String inferred = ImageFormats.inferFormatFromImageData(imageData);
        if (inferred != null && !inferred.equalsIgnoreCase(format)) {
            LOGGER.fine(String.format("Overriding tile format: was %s, set to %s", format, inferred));
        }
        return inferred != null ? inferred : format;
    }

    protected static BufferedImage readImage(byte[] data, String format) throws IOException {
        ByteArrayInputStream bis = new ByteArrayInputStream(data);
        Iterator<ImageReader> readers = ImageIO.getImageReadersByFormatName(TPKReader.getImageFormat(data, format));
        ImageReader reader = readers.next();
        try (ImageInputStream iis = ImageIO.createImageInputStream(bis);){
            reader.setInput(iis, true);
            ImageReadParam param = reader.getDefaultReadParam();
            BufferedImage bufferedImage = reader.read(0, param);
            return bufferedImage;
        }
    }

    protected BufferedImage getStartImage(BufferedImage copyFrom, int width, int height) {
        Hashtable<String, Object> properties = null;
        if (copyFrom.getPropertyNames() != null) {
            properties = new Hashtable<String, Object>();
            for (String name : copyFrom.getPropertyNames()) {
                properties.put(name, copyFrom.getProperty(name));
            }
        }
        SampleModel sm = copyFrom.getSampleModel().createCompatibleSampleModel(width, height);
        WritableRaster raster = Raster.createWritableRaster(sm, null);
        BufferedImage image = new BufferedImage(copyFrom.getColorModel(), raster, copyFrom.isAlphaPremultiplied(), properties);
        this.setBackground(image, new Color(0, true));
        return image;
    }

    protected BufferedImage getStartImage(int imageType, int width, int height) {
        if (imageType == 0) {
            imageType = 5;
        }
        BufferedImage image = new BufferedImage(width, height, imageType);
        this.setBackground(image, new Color(0, true));
        return image;
    }

    protected BufferedImage getStartImage(int width, int height) {
        return this.getStartImage(0, width, height);
    }

    protected void setBackground(BufferedImage image, Color bgColor) {
        Graphics2D g2D = (Graphics2D)image.getGraphics();
        Color save = g2D.getColor();
        g2D.setColor(bgColor);
        g2D.fillRect(0, 0, image.getWidth(), image.getHeight());
        g2D.setColor(save);
    }

    static {
        try {
            SPHERICAL_MERCATOR = CRS.decode((String)"EPSG:3857", (boolean)true);
            WGS_84 = CRS.decode((String)"EPSG:4326", (boolean)true);
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        WORLD_ENVELOPE = new ReferencedEnvelope(-2.003750834E7, 2.003750834E7, -2.003750834E7, 2.003750834E7, SPHERICAL_MERCATOR);
    }

    static class TileImage {
        long col;
        long row;
        BufferedImage image;

        TileImage(TPKTile tile) {
            this.col = tile.col;
            this.row = tile.row;
            if (tile.tileData != null && tile.tileData.length > 0) {
                try {
                    this.image = TPKReader.readImage(tile.tileData, tile.imageFormat);
                }
                catch (Exception ex) {
                    String template = "Bad tile data, zl=%d, row=%d, col=%d ==> %s";
                    LOGGER.info(String.format(template, tile.zoomLevel, this.row, this.col, ex.getMessage()));
                    this.image = null;
                }
            }
        }
    }

    public static enum ImageFormats {
        FMT_JPG("jpg", new byte[]{-1, -40}),
        FMT_PNG("png", new byte[]{-119, 80, 78, 71});

        private final String format;
        private final byte[] signature;

        private ImageFormats(String format, byte[] signature) {
            this.format = format;
            this.signature = signature;
        }

        public static String inferFormatFromImageData(byte[] imageData) {
            for (ImageFormats format : ImageFormats.values()) {
                boolean matches = true;
                try {
                    for (int index = 0; index < format.signature.length; ++index) {
                        if (imageData[index] == format.signature[index]) continue;
                        matches = false;
                        break;
                    }
                }
                catch (Exception ex) {
                    matches = false;
                }
                if (!matches) continue;
                return format.format;
            }
            return null;
        }
    }
}

