/*
 *    GeoTools - The Open Source Java GIS Toolkit
 *    http://geotools.org
 *
 *    (C) 2011, Open Source Geospatial Foundation (OSGeo)
 *    (C) 2004-2005, Open Geospatial Consortium Inc.
 *
 *    All Rights Reserved. http://www.opengis.org/legal/
 */
package org.opengis.metadata.identification;

import static org.opengis.annotation.Obligation.CONDITIONAL;
import static org.opengis.annotation.Specification.ISO_19115;

import java.util.ArrayList;
import java.util.List;
import org.opengis.annotation.UML;
import org.opengis.util.CodeList;

/**
 * Status of the dataset or progress of a review.
 *
 * @version <A HREF="http://www.opengeospatial.org/standards/as#01-111">ISO 19115</A>
 * @author Martin Desruisseaux (IRD)
 * @since GeoAPI 2.0
 */
@UML(identifier = "MD_ProgressCode", specification = ISO_19115)
public final class Progress extends CodeList<Progress> {
    /** Serial number for compatibility with different versions. */
    private static final long serialVersionUID = 7521085150853319219L;

    /** List of all enumerations of this type. Must be declared before any enum declaration. */
    private static final List<Progress> VALUES = new ArrayList<>(7);

    /** Production of the data has been completed. */
    @UML(identifier = "completed", obligation = CONDITIONAL, specification = ISO_19115)
    public static final Progress COMPLETED = new Progress("COMPLETED");

    /** Data has been stored in an offline storage facility */
    @UML(identifier = "historicalArchive", obligation = CONDITIONAL, specification = ISO_19115)
    public static final Progress HISTORICAL_ARCHIVE = new Progress("HISTORICAL_ARCHIVE");

    /** Data is no longer relevant. */
    @UML(identifier = "obsolete", obligation = CONDITIONAL, specification = ISO_19115)
    public static final Progress OBSOLETE = new Progress("OBSOLETE");

    /** Data is continually being updated. */
    @UML(identifier = "onGoing", obligation = CONDITIONAL, specification = ISO_19115)
    public static final Progress ON_GOING = new Progress("ON_GOING");

    /** Fixed date has been established upon or by which the data will be created or updated. */
    @UML(identifier = "planned", obligation = CONDITIONAL, specification = ISO_19115)
    public static final Progress PLANNED = new Progress("PLANNED");

    /** Data needs to be generated or updated. */
    @UML(identifier = "required", obligation = CONDITIONAL, specification = ISO_19115)
    public static final Progress REQUIRED = new Progress("REQUIRED");

    /** Data is currently in the process of being created. */
    @UML(identifier = "underDevelopment", obligation = CONDITIONAL, specification = ISO_19115)
    public static final Progress UNDER_DEVELOPMENT = new Progress("UNDER_DEVELOPMENT");

    /**
     * Constructs an enum with the given name. The new enum is automatically added to the list
     * returned by {@link #values}.
     *
     * @param name The enum name. This name must not be in use by an other enum of this type.
     */
    private Progress(final String name) {
        super(name, VALUES);
    }

    /**
     * Returns the list of {@code Progress}s.
     *
     * @return The list of codes declared in the current JVM.
     */
    public static Progress[] values() {
        synchronized (VALUES) {
            return VALUES.toArray(new Progress[VALUES.size()]);
        }
    }

    /** Returns the list of enumerations of the same kind than this enum. */
    @Override
    public Progress[] family() {
        return values();
    }

    /**
     * Returns the progress that matches the given string, or returns a new one if none match it.
     *
     * @param code The name of the code to fetch or to create.
     * @return A code matching the given name.
     */
    public static Progress valueOf(String code) {
        return valueOf(Progress.class, code);
    }
}
