/*
 *    GeoTools - The Open Source Java GIS Toolkit
 *    http://geotools.org
 *
 *    (C) 2011, Open Source Geospatial Foundation (OSGeo)
 *    (C) 2004-2005, Open Geospatial Consortium Inc.
 *
 *    All Rights Reserved. http://www.opengis.org/legal/
 */
package org.opengis.metadata.distribution;

import static org.opengis.annotation.ComplianceLevel.CORE;
import static org.opengis.annotation.Obligation.OPTIONAL;
import static org.opengis.annotation.Specification.ISO_19115;

import java.util.Collection;
import org.opengis.annotation.Profile;
import org.opengis.annotation.UML;
import org.opengis.metadata.citation.OnLineResource;
import org.opengis.util.InternationalString;

/**
 * Technical means and media by which a resource is obtained from the distributor.
 *
 * @version <A HREF="http://www.opengeospatial.org/standards/as#01-111">ISO 19115</A>
 * @author Martin Desruisseaux (IRD)
 * @author Cory Horner (Refractions Research)
 * @since GeoAPI 2.0
 */
@Profile(level = CORE)
@UML(identifier = "MD_DigitalTransferOptions", specification = ISO_19115)
public interface DigitalTransferOptions {
    /**
     * Tiles, layers, geographic areas, <cite>etc.</cite>, in which data is available.
     *
     * @return Tiles, layers, geographic areas, <cite>etc.</cite> in which data is available, or
     *     {@code null}.
     */
    @UML(identifier = "unitsOfDistribution", obligation = OPTIONAL, specification = ISO_19115)
    InternationalString getUnitsOfDistribution();

    /**
     * Estimated size of a unit in the specified transfer format, expressed in megabytes. The
     * transfer size is &gt; 0.0. Returns {@code null} if the transfer size is unknown.
     *
     * @return Estimated size of a unit in the specified transfer format in megabytes, or {@code
     *     null}.
     */
    @UML(identifier = "transferSize", obligation = OPTIONAL, specification = ISO_19115)
    Double getTransferSize();

    /**
     * Information about online sources from which the resource can be obtained.
     *
     * @return Online sources from which the resource can be obtained.
     */
    @Profile(level = CORE)
    @UML(identifier = "onLine", obligation = OPTIONAL, specification = ISO_19115)
    Collection<? extends OnLineResource> getOnLines();

    /**
     * Information about offline media on which the resource can be obtained.
     *
     * @return offline media on which the resource can be obtained, or {@code null}.
     */
    @UML(identifier = "offLine", obligation = OPTIONAL, specification = ISO_19115)
    Medium getOffLine();
}
