/*
 *    GeoTools - The Open Source Java GIS Toolkit
 *    http://geotools.org
 *
 *    (C) 2011, Open Source Geospatial Foundation (OSGeo)
 *    (C) 2003-2005, Open Geospatial Consortium Inc.
 *
 *    All Rights Reserved. http://www.opengis.org/legal/
 */
package org.opengis.geometry.coordinate;

import static org.opengis.annotation.Obligation.MANDATORY;
import static org.opengis.annotation.Specification.ISO_19107;

import java.util.List;
import org.opengis.annotation.UML;
import org.opengis.geometry.primitive.CurveInterpolation;
import org.opengis.geometry.primitive.CurveSegment;

/**
 * Sequence of geodesic segments. The interface essentially combines a <code>
 * Sequence&lt;{@link Geodesic}&gt;</code> into a single object, with the obvious savings of storage
 * space.
 *
 * @version <A HREF="http://www.opengeospatial.org/standards/as">ISO 19107</A>
 * @author Martin Desruisseaux (IRD)
 * @since GeoAPI 1.0
 * @see GeometryFactory#createGeodesicString
 */
@UML(identifier = "GM_GeodesicString", specification = ISO_19107)
public interface GeodesicString extends CurveSegment {
    /**
     * Returns a sequence of positions between which this {@code GeodesicString} is interpolated
     * using geodesics from the geoid or {@linkplain org.opengis.referencing.datum.Ellipsoid
     * ellipsoid} of the {@linkplain org.opengis.referencing.crs.CoordinateReferenceSystem
     * coordinate reference system} being used. The organization of these points is identical to
     * that in {@link LineString}.
     *
     * @return The control points.
     */
    @UML(identifier = "controlPoint", obligation = MANDATORY, specification = ISO_19107)
    PointArray getControlPoints();

    /**
     * The interpolation for a {@code GeodesicString} is "{@linkplain CurveInterpolation#GEODESIC
     * geodesic}".
     *
     * @return Always {@link CurveInterpolation#GEODESIC}.
     */
    @Override
    @UML(identifier = "interpolation", obligation = MANDATORY, specification = ISO_19107)
    CurveInterpolation getInterpolation();

    /**
     * Decomposes a geodesic string into an equivalent sequence of geodesic segments.
     *
     * @return The equivalent sequence of geodesic segments.
     */
    @UML(identifier = "asGM_Geodesic", obligation = MANDATORY, specification = ISO_19107)
    List<Geodesic> asGeodesics();
}
