/*
 *    GeoTools - The Open Source Java GIS Toolkit
 *    http://geotools.org
 *
 *    (C) 2004-2008, Open Source Geospatial Foundation (OSGeo)
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation;
 *    version 2.1 of the License.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 */
package org.geotools.metadata.iso;

import java.io.Serializable;
import org.geotools.metadata.InvalidMetadataException;
import org.geotools.metadata.MetadataStandard;
import org.geotools.metadata.ModifiableMetadata;
import org.geotools.metadata.i18n.ErrorKeys;
import org.geotools.metadata.i18n.Errors;

/**
 * A superclass for implementing ISO 19115 metadata interfaces. Subclasses must implement at least
 * one of the ISO MetaData interface provided by <A HREF="http://geoapi.sourceforge.net">GeoAPI</A>.
 *
 * @since 2.1
 * @version $Id$
 * @author Jody Garnett
 * @author Martin Desruisseaux
 */
public class MetadataEntity extends ModifiableMetadata implements Serializable {
    /** Serial number for interoperability with different versions. */
    private static final long serialVersionUID = 5730550742604669102L;

    /** Constructs an initially empty metadata entity. */
    protected MetadataEntity() {
        super();
    }

    /**
     * Constructs a metadata entity initialized with the values from the specified metadata. The
     * {@code source} metadata must implements the same metadata interface than this class.
     *
     * @param source The metadata to copy values from.
     * @throws ClassCastException if the specified metadata don't implements the expected metadata
     *     interface.
     * @since 2.4
     */
    protected MetadataEntity(final Object source) throws ClassCastException {
        super(source);
    }

    /**
     * Returns the metadata standard implemented by subclasses, which is {@linkplain
     * MetadataStandard#ISO_19115 ISO 19115}.
     *
     * @since 2.4
     */
    @Override
    public MetadataStandard getStandard() {
        return MetadataStandard.ISO_19115;
    }

    /**
     * Makes sure that an argument is non-null. This is used for checking if a mandatory attribute
     * is presents.
     *
     * @param name Argument name.
     * @param object User argument.
     * @throws InvalidMetadataException if {@code object} is null.
     * @since 2.4
     */
    protected static void ensureNonNull(final String name, final Object object)
            throws InvalidMetadataException {
        if (object == null) {
            throw new InvalidMetadataException(Errors.format(ErrorKeys.NULL_ATTRIBUTE_$1, name));
        }
    }
}
