/*
 * Decompiled with CFR 0.152.
 */
package org.geotools.renderer.crs;

import java.io.InputStreamReader;
import java.io.Reader;
import java.util.HashMap;
import java.util.List;
import java.util.zip.GZIPInputStream;
import org.geotools.api.referencing.FactoryException;
import org.geotools.api.referencing.NoSuchAuthorityCodeException;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;
import org.geotools.api.referencing.crs.SingleCRS;
import org.geotools.api.referencing.operation.MathTransform;
import org.geotools.api.referencing.operation.TransformException;
import org.geotools.geometry.jts.JTS;
import org.geotools.geometry.jts.LiteCoordinateSequence;
import org.geotools.geometry.jts.ReferencedEnvelope;
import org.geotools.referencing.CRS;
import org.geotools.referencing.crs.DefaultGeographicCRS;
import org.geotools.referencing.operation.projection.MapProjection;
import org.geotools.referencing.operation.projection.PolarStereographic;
import org.geotools.referencing.operation.transform.IdentityTransform;
import org.geotools.renderer.crs.ProjectionHandler;
import org.geotools.renderer.crs.ProjectionHandlerFinder;
import org.geotools.renderer.crs.WrappingProjectionHandler;
import org.geotools.util.Base64;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.CoordinateSequence;
import org.locationtech.jts.geom.Envelope;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryCollection;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.MultiLineString;
import org.locationtech.jts.geom.MultiPolygon;
import org.locationtech.jts.geom.Polygon;
import org.locationtech.jts.io.WKBReader;
import org.locationtech.jts.io.WKTReader;

public class ProjectionHandlerTest {
    static final double EPS = 1.0E-5;
    static CoordinateReferenceSystem WGS84;
    static CoordinateReferenceSystem ED50_LATLON;
    static CoordinateReferenceSystem UTM32N;
    static CoordinateReferenceSystem MERCATOR;
    static CoordinateReferenceSystem MERCATOR_SHIFTED;
    static CoordinateReferenceSystem ED50;
    static CoordinateReferenceSystem OSM;
    static GeometryFactory gf;

    @BeforeClass
    public static void setup() throws Exception {
        WGS84 = DefaultGeographicCRS.WGS84;
        UTM32N = CRS.decode((String)"EPSG:32632", (boolean)true);
        MERCATOR_SHIFTED = CRS.decode((String)"EPSG:3349", (boolean)true);
        MERCATOR = CRS.decode((String)"EPSG:3395", (boolean)true);
        OSM = CRS.decode((String)"EPSG:3857", (boolean)true);
        ED50 = CRS.decode((String)"EPSG:4230", (boolean)true);
        ED50_LATLON = CRS.decode((String)"urn:x-ogc:def:crs:EPSG:4230", (boolean)false);
        MapProjection.SKIP_SANITY_CHECKS = true;
    }

    @AfterClass
    public static void teardown() throws Exception {
        MapProjection.SKIP_SANITY_CHECKS = false;
    }

    @Test
    public void testWrappingOn3DCRS() throws Exception {
        CoordinateReferenceSystem crs = CRS.decode((String)"EPSG:4939", (boolean)true);
        SingleCRS hcrs = CRS.getHorizontalCRS((CoordinateReferenceSystem)crs);
        ReferencedEnvelope wgs84Envelope = new ReferencedEnvelope(-190.0, 60.0, -90.0, 45.0, (CoordinateReferenceSystem)hcrs);
        HashMap<String, Double> params = new HashMap<String, Double>();
        params.put("advancedProjectionDensify", 1.0);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)wgs84Envelope, (CoordinateReferenceSystem)crs, (boolean)true, params);
        Assert.assertNull((Object)handler.validAreaBounds);
        List envelopes = handler.getQueryEnvelopes();
        Assert.assertEquals((long)2L, (long)envelopes.size());
        ReferencedEnvelope expected = new ReferencedEnvelope(170.0, 180.0, -90.0, 45.0, (CoordinateReferenceSystem)hcrs);
        Assert.assertTrue((boolean)envelopes.remove(wgs84Envelope));
        Assert.assertEquals((Object)expected, envelopes.get(0));
    }

    @Test
    public void testWrappingDisabledHeuristic() throws Exception {
        ReferencedEnvelope world = new ReferencedEnvelope(-180.0, 180.0, -40.0, 40.0, WGS84);
        ReferencedEnvelope mercatorEnvelope = world.transform(MERCATOR, true);
        mercatorEnvelope.translate(mercatorEnvelope.getWidth() / 2.0, 0.0);
        Geometry g = new WKTReader().read("LINESTRING(-40 20, 190 20)");
        HashMap<String, Boolean> params = new HashMap<String, Boolean>();
        params.put("datelineWrappingCheckEnabled", false);
        MathTransform mt = CRS.findMathTransform((CoordinateReferenceSystem)WGS84, (CoordinateReferenceSystem)MERCATOR, (boolean)true);
        Geometry reprojected = JTS.transform((Geometry)g, (MathTransform)mt);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)mercatorEnvelope, (CoordinateReferenceSystem)WGS84, (boolean)true, params);
        Geometry processed = handler.postProcess(mt, reprojected);
        Assert.assertEquals((Object)processed.getGeometryN(0), (Object)reprojected);
        params.put("datelineWrappingCheckEnabled", true);
        handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)mercatorEnvelope, (CoordinateReferenceSystem)WGS84, (boolean)true, params);
        processed = handler.postProcess(mt, reprojected);
        Assert.assertNotEquals((Object)processed.getGeometryN(0), (Object)reprojected);
    }

    @Test
    public void testDensification() throws Exception {
        ReferencedEnvelope wgs84Envelope = new ReferencedEnvelope(-190.0, 60.0, -90.0, 45.0, WGS84);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)wgs84Envelope, (CoordinateReferenceSystem)WGS84, (boolean)true);
        LineString line = gf.createLineString(new Coordinate[]{new Coordinate(40.0, 45.0), new Coordinate(40.0, 88.0)});
        LineString notDensified = (LineString)handler.preProcess((Geometry)line);
        Assert.assertEquals((long)2L, (long)notDensified.getCoordinates().length);
        HashMap<String, Double> params = new HashMap<String, Double>();
        params.put("advancedProjectionDensify", 1.0);
        handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)wgs84Envelope, (CoordinateReferenceSystem)WGS84, (boolean)true, params);
        LineString densified = (LineString)handler.preProcess((Geometry)line);
        Assert.assertEquals((long)44L, (long)densified.getCoordinates().length);
    }

    @Test
    public void testDensificationOOM() throws Exception {
        ReferencedEnvelope wgs84Envelope = new ReferencedEnvelope(-190.0, 60.0, -90.0, 45.0, WGS84);
        HashMap<String, Double> params = new HashMap<String, Double>();
        params.put("advancedProjectionDensify", 1.0E-9);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)wgs84Envelope, (CoordinateReferenceSystem)WGS84, (boolean)true, params);
        LineString line = gf.createLineString(new Coordinate[]{new Coordinate(40.0, 45.0), new Coordinate(40.0, 85.0)});
        LineString densified = (LineString)handler.preProcess((Geometry)line);
        Assert.assertEquals((long)152589L, (long)densified.getCoordinates().length);
    }

    @Test
    public void testQueryWrappingWGS84() throws Exception {
        ReferencedEnvelope wgs84Envelope = new ReferencedEnvelope(-190.0, 60.0, -90.0, 45.0, WGS84);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)wgs84Envelope, (CoordinateReferenceSystem)WGS84, (boolean)true);
        Assert.assertNull((Object)handler.validAreaBounds);
        List envelopes = handler.getQueryEnvelopes();
        Assert.assertEquals((long)2L, (long)envelopes.size());
        ReferencedEnvelope expected = new ReferencedEnvelope(170.0, 180.0, -90.0, 45.0, WGS84);
        Assert.assertTrue((boolean)envelopes.remove(wgs84Envelope));
        Assert.assertEquals((Object)expected, envelopes.get(0));
    }

    @Test
    public void testQueryWrappingED50LatLon() throws Exception {
        ReferencedEnvelope envelope = new ReferencedEnvelope(-90.0, 45.0, -190.0, 60.0, ED50_LATLON);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)envelope, (CoordinateReferenceSystem)ED50_LATLON, (boolean)true);
        Assert.assertNull((Object)handler.validAreaBounds);
        List envelopes = handler.getQueryEnvelopes();
        Assert.assertEquals((long)2L, (long)envelopes.size());
        ReferencedEnvelope expected = new ReferencedEnvelope(-90.0, 45.0, 170.0, 180.0, ED50_LATLON);
        Assert.assertTrue((boolean)envelopes.remove(envelope));
        Assert.assertEquals((Object)expected, envelopes.get(0));
    }

    @Test
    public void testValidAreaMercator() throws Exception {
        ReferencedEnvelope world = new ReferencedEnvelope(-180.0, 180.0, -89.9999, 89.9999, WGS84);
        ReferencedEnvelope mercatorEnvelope = world.transform(MERCATOR_SHIFTED, true);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)mercatorEnvelope, (CoordinateReferenceSystem)WGS84, (boolean)true);
        ReferencedEnvelope va = handler.validAreaBounds;
        Assert.assertNotNull((Object)va);
        Assert.assertTrue((va.getMinX() <= -180.0 ? 1 : 0) != 0);
        Assert.assertTrue((va.getMaxX() >= 180.0 ? 1 : 0) != 0);
        Assert.assertTrue((-90.0 < va.getMinY() ? 1 : 0) != 0);
        Assert.assertTrue((90.0 > va.getMaxY() ? 1 : 0) != 0);
    }

    @Test
    public void testValidAreaLambertAzimuthalEqualArea() throws Exception {
        ReferencedEnvelope wgs84north = new ReferencedEnvelope(-120.0, 0.0, 45.0, 90.0, WGS84);
        ReferencedEnvelope laeNorth = wgs84north.transform(CRS.decode((String)"EPSG:3408"), true);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)laeNorth, (CoordinateReferenceSystem)WGS84, (boolean)true);
        ReferencedEnvelope va = handler.validAreaBounds;
        Assert.assertNotNull((Object)va);
        Assert.assertEquals((Object)va.getCoordinateReferenceSystem(), (Object)WGS84);
        Assert.assertEquals((double)-180.0, (double)va.getMinX(), (double)0.0);
        Assert.assertEquals((double)180.0, (double)va.getMaxX(), (double)0.0);
        Assert.assertEquals((double)0.0, (double)va.getMinY(), (double)0.0);
        Assert.assertEquals((double)90.0, (double)va.getMaxY(), (double)0.0);
        ReferencedEnvelope wgs84South = new ReferencedEnvelope(-120.0, 0.0, -90.0, -45.0, WGS84);
        ReferencedEnvelope laeSouth = wgs84South.transform(CRS.decode((String)"EPSG:3409"), true);
        handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)laeSouth, (CoordinateReferenceSystem)WGS84, (boolean)true);
        va = handler.validAreaBounds;
        Assert.assertNotNull((Object)va);
        Assert.assertEquals((double)-180.0, (double)va.getMinX(), (double)0.0);
        Assert.assertEquals((double)180.0, (double)va.getMaxX(), (double)0.0);
        Assert.assertEquals((double)-90.0, (double)va.getMinY(), (double)0.0);
        Assert.assertEquals((double)0.0, (double)va.getMaxY(), (double)0.0);
    }

    @Test
    public void testValidAreaWorldVanDerGrinten() throws Exception {
        String wkt = "PROJCS[\"World_Van_der_Grinten_I\", \n  GEOGCS[\"GCS_WGS_1984\", \n    DATUM[\"D_WGS_1984\", \n      SPHEROID[\"WGS_1984\", 6378137.0, 298.257223563]], \n    PRIMEM[\"Greenwich\", 0.0], \n    UNIT[\"degree\", 0.017453292519943295], \n    AXIS[\"Longitude\", EAST], \n    AXIS[\"Latitude\", NORTH]], \n  PROJECTION[\"World_Van_der_Grinten_I\"], \n  PARAMETER[\"central_meridian\", 0.0], \n  PARAMETER[\"false_easting\", 0.0], \n  PARAMETER[\"false_northing\", 0.0], \n  UNIT[\"m\", 1.0], \n  AXIS[\"x\", EAST], \n  AXIS[\"y\", NORTH], \n  AUTHORITY[\"EPSG\",\"54029\"]]";
        ReferencedEnvelope envelopeWgs84 = new ReferencedEnvelope(-180.0, 180.0, -90.0, 90.0, WGS84);
        ReferencedEnvelope envelope = envelopeWgs84.transform(CRS.parseWKT((String)wkt), true);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)envelope, (CoordinateReferenceSystem)WGS84, (boolean)true);
        ReferencedEnvelope va = handler.validAreaBounds;
        Assert.assertNotNull((Object)va);
        Assert.assertEquals((Object)va.getCoordinateReferenceSystem(), (Object)WGS84);
        Assert.assertEquals((double)-2.147483647E9, (double)va.getMinX(), (double)0.0);
        Assert.assertEquals((double)2.147483647E9, (double)va.getMaxX(), (double)0.0);
        Assert.assertEquals((double)-90.0, (double)va.getMinY(), (double)0.0);
        Assert.assertEquals((double)90.0, (double)va.getMaxY(), (double)0.0);
    }

    @Test
    public void testValidAreaLambertConformal() throws Exception {
        ReferencedEnvelope wgs84north = new ReferencedEnvelope(-120.0, 0.0, 45.0, 90.0, WGS84);
        ReferencedEnvelope laeNorth = wgs84north.transform(CRS.decode((String)"EPSG:2062"), true);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)laeNorth, (CoordinateReferenceSystem)WGS84, (boolean)true);
        ReferencedEnvelope va = handler.validAreaBounds;
        Assert.assertNotNull((Object)va);
        Assert.assertEquals((double)-179.9, (double)va.getMinX(), (double)0.0);
        Assert.assertEquals((double)179.9, (double)va.getMaxX(), (double)0.0);
        Assert.assertEquals((double)-4.0, (double)va.getMinY(), (double)0.0);
        Assert.assertEquals((double)90.0, (double)va.getMaxY(), (double)0.0);
        ReferencedEnvelope wgs84South = new ReferencedEnvelope(-180.0, -90.0, -40.0, 0.0, WGS84);
        ReferencedEnvelope laeSouth = wgs84South.transform(CRS.decode((String)"EPSG:2194"), true);
        handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)laeSouth, (CoordinateReferenceSystem)WGS84, (boolean)true);
        va = handler.validAreaBounds;
        Assert.assertNotNull((Object)va);
        Assert.assertEquals((double)-180.0, (double)va.getMinX(), (double)0.0);
        Assert.assertEquals((double)180.0, (double)va.getMaxX(), (double)0.0);
        Assert.assertEquals((double)-90.0, (double)va.getMinY(), (double)0.0);
        Assert.assertEquals((double)29.73, (double)va.getMaxY(), (double)0.01);
    }

    @Test
    public void testCutGeometryLambertConformal() throws Exception {
        ReferencedEnvelope wgs84South = new ReferencedEnvelope(-180.0, -90.0, -40.0, 0.0, WGS84);
        ReferencedEnvelope laeSouth = wgs84South.transform(CRS.decode((String)"EPSG:2194"), true);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)laeSouth, (CoordinateReferenceSystem)WGS84, (boolean)true);
        Polygon geometry = JTS.toGeometry((Envelope)new Envelope(5.0, 15.0, 0.0, 10.0));
        Geometry preProcessed = handler.preProcess((Geometry)geometry);
        Assert.assertTrue((String)"Should have sliced the geometry in two parts", (boolean)(preProcessed instanceof MultiPolygon));
    }

    @Test
    public void testRobustCutting() throws Exception {
        Geometry geom;
        String wkt = "PROJCS[\"Asia_South_Lambert_Conformal_Conic\", \n  GEOGCS[\"GCS_WGS_1984\", \n    DATUM[\"WGS_1984\", \n      SPHEROID[\"WGS_1984\", 6378137.0, 298.257223563]], \n    PRIMEM[\"Greenwich\", 0.0], \n    UNIT[\"degree\", 0.017453292519943295], \n    AXIS[\"Longitude\", EAST], \n    AXIS[\"Latitude\", NORTH]], \n  PROJECTION[\"Lambert_Conformal_Conic_2SP\"], \n  PARAMETER[\"central_meridian\", 125.0], \n  PARAMETER[\"latitude_of_origin\", -15.0], \n  PARAMETER[\"standard_parallel_1\", 7.0], \n  PARAMETER[\"false_easting\", 0.0], \n  PARAMETER[\"false_northing\", 0.0], \n  PARAMETER[\"scale_factor\", 1.0], \n  PARAMETER[\"standard_parallel_2\", -32.0], \n  UNIT[\"m\", 1.0], \n  AXIS[\"x\", EAST], \n  AXIS[\"y\", NORTH], \n  AUTHORITY[\"EPSG\",\"102030\"]]";
        CoordinateReferenceSystem crs = CRS.parseWKT((String)wkt);
        try (InputStreamReader reader = new InputStreamReader(new GZIPInputStream(ProjectionHandlerTest.class.getResourceAsStream("para.wkt.gz")));){
            geom = new WKTReader().read((Reader)reader);
        }
        ReferencedEnvelope re = new ReferencedEnvelope(1.2248782489837505E7, 2.0320948299686E7, -4848266.752703998, 3223899.0571445003, crs);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)re, (CoordinateReferenceSystem)WGS84, (boolean)true);
        Geometry preprocessed = handler.preProcess(geom);
        Assert.assertNotEquals((Object)preprocessed, (Object)geom);
    }

    @Test
    public void testQueryWrappingMercatorWorld() throws Exception {
        ReferencedEnvelope world = new ReferencedEnvelope(-200.0, 200.0, -89.0, 89.0, WGS84);
        ReferencedEnvelope mercatorEnvelope = world.transform(MERCATOR_SHIFTED, true);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)mercatorEnvelope, (CoordinateReferenceSystem)WGS84, (boolean)true);
        List envelopes = handler.getQueryEnvelopes();
        Assert.assertEquals((long)1L, (long)envelopes.size());
        ReferencedEnvelope env = (ReferencedEnvelope)envelopes.get(0);
        Assert.assertEquals((double)-180.0, (double)env.getMinX(), (double)1.0E-5);
        Assert.assertEquals((double)180.0, (double)env.getMaxX(), (double)1.0E-5);
        Assert.assertEquals((double)-85.0, (double)env.getMinY(), (double)0.1);
        Assert.assertEquals((double)85.0, (double)env.getMaxY(), (double)0.1);
    }

    @Test
    public void testQueryOutsideValidArea() throws Exception {
        ReferencedEnvelope world = new ReferencedEnvelope(-200.0, 200.0, -89.0, -86.0, WGS84);
        ReferencedEnvelope mercatorEnvelope = world.transform(MERCATOR_SHIFTED, true);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)mercatorEnvelope, (CoordinateReferenceSystem)WGS84, (boolean)true);
        List envelopes = handler.getQueryEnvelopes();
        Assert.assertEquals((long)0L, (long)envelopes.size());
    }

    @Test
    public void testQueryWrappingMercatorSeparate() throws Exception {
        ReferencedEnvelope world = new ReferencedEnvelope(160.0, 180.0, -40.0, 40.0, WGS84);
        ReferencedEnvelope mercatorEnvelope = world.transform(MERCATOR, true);
        mercatorEnvelope.translate(mercatorEnvelope.getWidth() / 2.0, 0.0);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)mercatorEnvelope, (CoordinateReferenceSystem)WGS84, (boolean)true);
        List envelopes = handler.getQueryEnvelopes();
        Assert.assertEquals((long)2L, (long)envelopes.size());
        ReferencedEnvelope reOrig = (ReferencedEnvelope)envelopes.get(0);
        Assert.assertEquals((double)170.0, (double)reOrig.getMinX(), (double)1.0E-5);
        Assert.assertEquals((double)180.0, (double)reOrig.getMaxX(), (double)1.0E-5);
        ReferencedEnvelope reAdded = (ReferencedEnvelope)envelopes.get(1);
        Assert.assertEquals((double)-180.0, (double)reAdded.getMinX(), (double)1.0E-5);
        Assert.assertEquals((double)-170.0, (double)reAdded.getMaxX(), (double)1.0E-5);
    }

    @Test
    public void testQueryWrappingPacificMercator() throws Exception {
        CoordinateReferenceSystem crs = CRS.decode((String)"EPSG:3832");
        ReferencedEnvelope mercatorEnvelope = new ReferencedEnvelope(1113195.0, 5565975.0, 0.0, 4838471.0, crs);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)mercatorEnvelope, (CoordinateReferenceSystem)WGS84, (boolean)true);
        List envelopes = handler.getQueryEnvelopes();
        Assert.assertEquals((long)2L, (long)envelopes.size());
        ReferencedEnvelope reOrig = (ReferencedEnvelope)envelopes.get(0);
        Assert.assertEquals((double)160.0, (double)reOrig.getMinX(), (double)1.0E-5);
        Assert.assertEquals((double)180.0, (double)reOrig.getMaxX(), (double)1.0E-5);
        ReferencedEnvelope reAdded = (ReferencedEnvelope)envelopes.get(1);
        Assert.assertEquals((double)-180.0, (double)reAdded.getMinX(), (double)1.0E-5);
        Assert.assertEquals((double)-160.0, (double)reAdded.getMaxX(), (double)1.0E-5);
    }

    @Test
    public void testValidAreaUTM() throws Exception {
        ReferencedEnvelope wgs84Envelope = new ReferencedEnvelope(8.0, 10.0, 40.0, 45.0, WGS84);
        ReferencedEnvelope utmEnvelope = wgs84Envelope.transform(UTM32N, true);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)utmEnvelope, (CoordinateReferenceSystem)WGS84, (boolean)true);
        ReferencedEnvelope va = handler.validAreaBounds;
        Assert.assertNotNull((Object)va);
        Assert.assertTrue((-81.0 < va.getMinX() && va.getMinX() <= 6.0 ? 1 : 0) != 0);
        Assert.assertTrue((12.0 <= va.getMaxX() && va.getMaxX() < 99.0 ? 1 : 0) != 0);
        Assert.assertEquals((double)-85.0, (double)va.getMinY(), (double)1.0E-5);
        Assert.assertEquals((double)85.0, (double)va.getMaxY(), (double)1.0E-5);
    }

    @Test
    public void testQueryUTM() throws Exception {
        ReferencedEnvelope wgs84Envelope = new ReferencedEnvelope(8.0, 10.0, 40.0, 45.0, WGS84);
        ReferencedEnvelope utmEnvelope = wgs84Envelope.transform(UTM32N, true);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)utmEnvelope, (CoordinateReferenceSystem)WGS84, (boolean)true);
        ReferencedEnvelope expected = utmEnvelope.transform(WGS84, true);
        List envelopes = handler.getQueryEnvelopes();
        Assert.assertEquals((long)1L, (long)envelopes.size());
        Assert.assertEquals((Object)expected, envelopes.get(0));
    }

    @Test
    public void testWrapGeometryMercator() throws Exception {
        ReferencedEnvelope world = new ReferencedEnvelope(160.0, 180.0, -40.0, 40.0, WGS84);
        ReferencedEnvelope mercatorEnvelope = world.transform(MERCATOR, true);
        mercatorEnvelope.translate(mercatorEnvelope.getWidth() / 2.0, 0.0);
        Geometry g = new WKTReader().read("LINESTRING(170 -40, 190 40)");
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)mercatorEnvelope, (CoordinateReferenceSystem)WGS84, (boolean)true);
        Assert.assertTrue((boolean)handler.requiresProcessing(g));
        Geometry preProcessed = handler.preProcess(g);
        Assert.assertEquals((Object)g, (Object)preProcessed);
        MathTransform mt = CRS.findMathTransform((CoordinateReferenceSystem)WGS84, (CoordinateReferenceSystem)MERCATOR, (boolean)true);
        Geometry transformed = JTS.transform((Geometry)g, (MathTransform)mt);
        Geometry postProcessed = handler.postProcess(mt.inverse(), transformed);
        Envelope env = postProcessed.getEnvelopeInternal();
        Assert.assertEquals((double)mercatorEnvelope.getMinX(), (double)env.getMinX(), (double)1.0E-5);
        Assert.assertEquals((double)mercatorEnvelope.getMaxX(), (double)env.getMaxX(), (double)1.0E-5);
    }

    @Test
    public void testXymGeometriesMeasuresArePreserved() throws Exception {
        Geometry geometry = new WKTReader().read("LINESTRINGM(170 -40 2, 190 40 7)");
        MathTransform transform = CRS.findMathTransform((CoordinateReferenceSystem)WGS84, (CoordinateReferenceSystem)MERCATOR, (boolean)true);
        Geometry transformed = JTS.transform((Geometry)geometry, (MathTransform)transform);
        MatcherAssert.assertThat((Object)transformed, (Matcher)Matchers.instanceOf(LineString.class));
        LineString line = (LineString)transformed;
        MatcherAssert.assertThat((Object)line.getCoordinateSequence().getDimension(), (Matcher)Matchers.is((Object)3));
        MatcherAssert.assertThat((Object)line.getCoordinateSequence().getMeasures(), (Matcher)Matchers.is((Object)1));
        MatcherAssert.assertThat((Object)line.getCoordinateSequence().getX(0), (Matcher)Matchers.closeTo((double)1.8924313434856504E7, (double)1.0E-5));
        MatcherAssert.assertThat((Object)line.getCoordinateSequence().getY(0), (Matcher)Matchers.closeTo((double)-4838471.398061137, (double)1.0E-5));
        MatcherAssert.assertThat((Object)line.getCoordinateSequence().getZ(0), (Matcher)Matchers.is((Object)Double.NaN));
        MatcherAssert.assertThat((Object)line.getCoordinateSequence().getM(0), (Matcher)Matchers.is((Object)2.0));
        MatcherAssert.assertThat((Object)line.getCoordinateSequence().getX(1), (Matcher)Matchers.closeTo((double)2.115070325072198E7, (double)1.0E-5));
        MatcherAssert.assertThat((Object)line.getCoordinateSequence().getY(1), (Matcher)Matchers.closeTo((double)4838471.398061137, (double)1.0E-5));
        MatcherAssert.assertThat((Object)line.getCoordinateSequence().getZ(1), (Matcher)Matchers.is((Object)Double.NaN));
        MatcherAssert.assertThat((Object)line.getCoordinateSequence().getM(1), (Matcher)Matchers.is((Object)7.0));
    }

    @Test
    public void testXyzmGeometriesMeasuresArePreserved() throws Exception {
        Geometry geometry = new WKTReader().read("LINESTRINGZM(170 -40 10 2, 190 40 15 7)");
        MathTransform transform = CRS.findMathTransform((CoordinateReferenceSystem)WGS84, (CoordinateReferenceSystem)MERCATOR, (boolean)true);
        Geometry transformed = JTS.transform((Geometry)geometry, (MathTransform)transform);
        MatcherAssert.assertThat((Object)transformed, (Matcher)Matchers.instanceOf(LineString.class));
        LineString line = (LineString)transformed;
        MatcherAssert.assertThat((Object)line.getCoordinateSequence().getDimension(), (Matcher)Matchers.is((Object)4));
        MatcherAssert.assertThat((Object)line.getCoordinateSequence().getMeasures(), (Matcher)Matchers.is((Object)1));
        MatcherAssert.assertThat((Object)line.getCoordinateSequence().getX(0), (Matcher)Matchers.closeTo((double)1.8924313434856504E7, (double)1.0E-5));
        MatcherAssert.assertThat((Object)line.getCoordinateSequence().getY(0), (Matcher)Matchers.closeTo((double)-4838471.398061137, (double)1.0E-5));
        MatcherAssert.assertThat((Object)line.getCoordinateSequence().getZ(0), (Matcher)Matchers.is((Object)10.0));
        MatcherAssert.assertThat((Object)line.getCoordinateSequence().getM(0), (Matcher)Matchers.is((Object)2.0));
        MatcherAssert.assertThat((Object)line.getCoordinateSequence().getX(1), (Matcher)Matchers.closeTo((double)2.115070325072198E7, (double)1.0E-5));
        MatcherAssert.assertThat((Object)line.getCoordinateSequence().getY(1), (Matcher)Matchers.closeTo((double)4838471.398061137, (double)1.0E-5));
        MatcherAssert.assertThat((Object)line.getCoordinateSequence().getZ(1), (Matcher)Matchers.is((Object)15.0));
        MatcherAssert.assertThat((Object)line.getCoordinateSequence().getM(1), (Matcher)Matchers.is((Object)7.0));
    }

    @Test
    public void testWrapGeometrySmall() throws Exception {
        CoordinateReferenceSystem FIJI = CRS.decode((String)"EPSG:3460", (boolean)true);
        Geometry g = new WKTReader().read("POLYGON ((2139122 5880020, 2139122 5880030, 2139922 5880030, 2139122 5880020))");
        Geometry original = g.copy();
        ReferencedEnvelope world = new ReferencedEnvelope(178.0, 181.0, -1.0, 1.0, WGS84);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)world, (CoordinateReferenceSystem)FIJI, (boolean)true);
        Assert.assertTrue((boolean)handler.requiresProcessing(g));
        Geometry preProcessed = handler.preProcess(g);
        Assert.assertEquals((Object)original, (Object)preProcessed);
        MathTransform mt = CRS.findMathTransform((CoordinateReferenceSystem)FIJI, (CoordinateReferenceSystem)WGS84);
        Geometry transformed = JTS.transform((Geometry)g, (MathTransform)mt);
        Geometry postProcessed = handler.postProcess(mt.inverse(), transformed);
        Assert.assertTrue((boolean)world.contains(postProcessed.getEnvelopeInternal()));
    }

    @Test
    public void testWorldLargeGeometry() throws Exception {
        ReferencedEnvelope world = new ReferencedEnvelope(-180.0, 180.0, -90.0, 90.0, WGS84);
        Geometry g = new WKTReader().read("POLYGON((-178 -90, -178 90, 178 90, 178 -90, -178 -90))");
        Geometry original = new WKTReader().read("POLYGON((-178 -90, -178 90, 178 90, 178 -90, -178 -90))");
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)world, (CoordinateReferenceSystem)WGS84, (boolean)true);
        Assert.assertTrue((boolean)handler.requiresProcessing(g));
        Geometry preProcessed = handler.preProcess(g);
        Assert.assertEquals((Object)original, (Object)preProcessed);
        Geometry postProcessed = handler.postProcess(CRS.findMathTransform((CoordinateReferenceSystem)WGS84, (CoordinateReferenceSystem)WGS84), g);
        Assert.assertEquals((Object)original, (Object)postProcessed);
    }

    @Test
    public void testWrapGeometryLatLonMultipleTimes() throws Exception {
        ReferencedEnvelope renderingEnvelope = new ReferencedEnvelope(-90.0, 90.0, -580.0, 540.0, ED50_LATLON);
        Geometry g = new WKTReader().read("POLYGON((-74 -33, -29 -33, -29 5, -74 5, -74 -33))");
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)renderingEnvelope, (CoordinateReferenceSystem)WGS84, (boolean)true);
        Assert.assertTrue((boolean)handler.requiresProcessing(g));
        Geometry preProcessed = handler.preProcess(g);
        MathTransform mt = handler.getRenderingTransform(CRS.findMathTransform((CoordinateReferenceSystem)WGS84, (CoordinateReferenceSystem)ED50_LATLON));
        Geometry transformed = JTS.transform((Geometry)preProcessed, (MathTransform)mt);
        Geometry postProcessed = handler.postProcess(mt, transformed);
        Assert.assertTrue((boolean)postProcessed.isValid());
        Assert.assertEquals((long)3L, (long)postProcessed.getNumGeometries());
    }

    @Test
    public void testWrapGeometryReprojectToLatLonED50() throws Exception {
        ReferencedEnvelope world = new ReferencedEnvelope(-80.0, 80.0, -180.0, 180.0, ED50_LATLON);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)world, (CoordinateReferenceSystem)WGS84, (boolean)true);
        String wkt = "POLYGON((178 -80, 178 80, 182 80, 182 80, 178 -80))";
        Geometry g = new WKTReader().read(wkt);
        Geometry original = new WKTReader().read(wkt);
        MathTransform mt = CRS.findMathTransform((CoordinateReferenceSystem)WGS84, (CoordinateReferenceSystem)ED50_LATLON);
        MathTransform prepared = handler.getRenderingTransform(mt);
        Geometry reprojected = JTS.transform((Geometry)original, (MathTransform)prepared);
        Assert.assertTrue((boolean)handler.requiresProcessing(g));
        Geometry preProcessed = handler.preProcess(g);
        Assert.assertEquals((Object)original, (Object)preProcessed);
        Geometry postProcessed = handler.postProcess(prepared, reprojected);
        Assert.assertTrue((boolean)(postProcessed instanceof MultiPolygon));
    }

    @Test
    public void testWrapAnctartica() throws Exception {
        ReferencedEnvelope world = new ReferencedEnvelope(-80.0, 80.0, -180.0, 180.0, ED50_LATLON);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)world, (CoordinateReferenceSystem)WGS84, (boolean)true);
        String wkt = "POLYGON((180 -90, 180 90, -180 90, -180 -90, 180 -90))";
        Geometry g = new WKTReader().read(wkt);
        MathTransform mt = CRS.findMathTransform((CoordinateReferenceSystem)WGS84, (CoordinateReferenceSystem)ED50_LATLON);
        MathTransform prepared = handler.getRenderingTransform(mt);
        Assert.assertTrue((boolean)handler.requiresProcessing(g));
        Geometry preProcessed = handler.preProcess(g);
        Geometry reprojected = JTS.transform((Geometry)preProcessed, (MathTransform)prepared);
        Assert.assertTrue((boolean)reprojected.isValid());
        reprojected.apply(coord -> {
            Assert.assertEquals((double)90.0, (double)Math.abs(coord.getOrdinate(0)), (double)0.1);
            Assert.assertEquals((double)180.0, (double)Math.abs(coord.getOrdinate(1)), (double)5.0);
        });
        Geometry postProcessed = handler.postProcess(prepared, reprojected);
        MatcherAssert.assertThat((Object)postProcessed, (Matcher)CoreMatchers.instanceOf(MultiPolygon.class));
        Assert.assertEquals((long)2L, (long)postProcessed.getNumGeometries());
    }

    @Test
    public void testWrapGeometryReprojectToED50() throws Exception {
        ReferencedEnvelope world = new ReferencedEnvelope(-80.0, 80.0, -180.0, 180.0, ED50);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)world, (CoordinateReferenceSystem)WGS84, (boolean)true);
        String wkt = "POLYGON((178 -80, 178 80, 182 80, 182 80, 178 -80))";
        Geometry g = new WKTReader().read(wkt);
        Geometry original = new WKTReader().read(wkt);
        MathTransform mt = CRS.findMathTransform((CoordinateReferenceSystem)WGS84, (CoordinateReferenceSystem)ED50);
        mt = handler.getRenderingTransform(mt);
        Geometry reprojected = JTS.transform((Geometry)original, (MathTransform)mt);
        Assert.assertTrue((boolean)handler.requiresProcessing(g));
        Geometry preProcessed = handler.preProcess(g);
        Assert.assertEquals((Object)original, (Object)preProcessed);
        Geometry postProcessed = handler.postProcess(mt, reprojected);
        Assert.assertTrue((boolean)(postProcessed instanceof MultiPolygon));
    }

    @Test
    public void testWrapJumpLast() throws Exception {
        ReferencedEnvelope world = new ReferencedEnvelope(-180.0, 180.0, -90.0, 90.0, WGS84);
        Geometry g = new WKTReader().read("POLYGON((-131 -73.5,0 -90,163 -60,174 -60,-131 -73.5))");
        Geometry original = new WKTReader().read("POLYGON((-131 -73.5,0 -90,163 -60,174 -60,-131 -73.5))");
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)world, (CoordinateReferenceSystem)WGS84, (boolean)true);
        Assert.assertTrue((boolean)handler.requiresProcessing(g));
        Geometry preProcessed = handler.preProcess(g);
        Assert.assertEquals((Object)original, (Object)preProcessed);
        Geometry postProcessed = handler.postProcess(CRS.findMathTransform((CoordinateReferenceSystem)WGS84, (CoordinateReferenceSystem)WGS84), g);
        Assert.assertEquals((Object)original, (Object)postProcessed);
    }

    @Test
    public void testWrapGeometryWGS84Duplicate() throws Exception {
        ReferencedEnvelope world = new ReferencedEnvelope(-200.0, 200.0, -90.0, 90.0, WGS84);
        Geometry g = new WKTReader().read("POLYGON((-178 -90, -178 90, 178 90, 178 -90, -178 -90))");
        Geometry original = new WKTReader().read("POLYGON((-178 -90, -178 90, 178 90, 178 -90, -178 -90))");
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)world, (CoordinateReferenceSystem)WGS84, (boolean)true);
        Assert.assertTrue((boolean)handler.requiresProcessing(g));
        Geometry preProcessed = handler.preProcess(g);
        Assert.assertEquals((Object)original, (Object)preProcessed);
        Geometry postProcessed = handler.postProcess(null, g);
        Envelope ppEnvelope = postProcessed.getEnvelopeInternal();
        Envelope expected = new Envelope(-538.0, 538.0, -90.0, 90.0);
        Assert.assertEquals((Object)expected, (Object)ppEnvelope);
    }

    @Test
    public void testDuplicateGeometryMercator() throws Exception {
        ReferencedEnvelope world = new ReferencedEnvelope(-180.0, 180.0, -50.0, 50.0, WGS84);
        ReferencedEnvelope mercatorEnvelope = world.transform(MERCATOR, true);
        Geometry g = new WKTReader().read("LINESTRING(170 -50, 190 50)");
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)mercatorEnvelope, (CoordinateReferenceSystem)WGS84, (boolean)true);
        Assert.assertTrue((boolean)handler.requiresProcessing(g));
        Geometry preProcessed = handler.preProcess(g);
        Assert.assertEquals((Object)g, (Object)preProcessed);
        MathTransform mt = CRS.findMathTransform((CoordinateReferenceSystem)WGS84, (CoordinateReferenceSystem)MERCATOR, (boolean)true);
        Geometry transformed = JTS.transform((Geometry)g, (MathTransform)mt);
        Geometry postProcessed = handler.postProcess(mt, transformed);
        Assert.assertTrue((boolean)(postProcessed instanceof MultiLineString));
        MultiLineString mls = (MultiLineString)postProcessed;
        Assert.assertEquals((long)2L, (long)mls.getNumGeometries());
        double twentyDegWidth = mercatorEnvelope.getWidth() / 18.0;
        Assert.assertEquals((double)twentyDegWidth, (double)mls.getGeometryN(0).getEnvelopeInternal().getWidth(), (double)1.0E-5);
        Assert.assertEquals((double)twentyDegWidth, (double)mls.getGeometryN(1).getEnvelopeInternal().getWidth(), (double)1.0E-5);
    }

    @Test
    public void testLimitExcessiveDuplication() throws Exception {
        ReferencedEnvelope renderingEnvelope = new ReferencedEnvelope(-1800000.0, 1800000.0, -50.0, 50.0, WGS84);
        Geometry g = new WKTReader().read("LINESTRING(-179 -89, 179 89)");
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)renderingEnvelope, (CoordinateReferenceSystem)WGS84, (boolean)true);
        Assert.assertTrue((boolean)handler.requiresProcessing(g));
        Geometry preProcessed = handler.preProcess(g);
        Assert.assertEquals((Object)g, (Object)preProcessed);
        Geometry postProcessed = handler.postProcess((MathTransform)IdentityTransform.create((int)2), g);
        Assert.assertTrue((boolean)(postProcessed instanceof MultiLineString));
        MultiLineString mls = (MultiLineString)postProcessed;
        Assert.assertEquals((long)(ProjectionHandlerFinder.WRAP_LIMIT * 2 + 1), (long)mls.getNumGeometries());
    }

    @Test
    public void testCutGeometryUTM() throws Exception {
        ReferencedEnvelope wgs84Envelope = new ReferencedEnvelope(8.0, 10.0, 40.0, 45.0, WGS84);
        ReferencedEnvelope utmEnvelope = wgs84Envelope.transform(UTM32N, true);
        Geometry g = new WKTReader().read("LINESTRING(-170 -40, 170 40)");
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)utmEnvelope, (CoordinateReferenceSystem)WGS84, (boolean)true);
        Assert.assertTrue((boolean)handler.requiresProcessing(g));
        Geometry preProcessed = handler.preProcess(g);
        Assert.assertFalse((boolean)preProcessed.equalsTopo(g));
        Assert.assertTrue((boolean)handler.validAreaBounds.contains(preProcessed.getEnvelopeInternal()));
    }

    @Test
    public void testPolarStereographic() throws Exception {
        ReferencedEnvelope envelope = new ReferencedEnvelope(-1.07E7, 1.47E7, -1.07E7, 1.47E7, CRS.decode((String)"EPSG:5041", (boolean)true));
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)envelope, (CoordinateReferenceSystem)WGS84, (boolean)true);
        Assert.assertNotNull((Object)handler);
        Assert.assertEquals((Object)envelope, (Object)handler.getRenderingEnvelope());
        Assert.assertTrue((boolean)(CRS.getMapProjection((CoordinateReferenceSystem)envelope.getCoordinateReferenceSystem()) instanceof PolarStereographic));
    }

    @Test
    public void testSkipInvalidGeometries() throws Exception {
        ReferencedEnvelope world = new ReferencedEnvelope(160.0, 180.0, -40.0, 40.0, WGS84);
        ReferencedEnvelope mercatorEnvelope = world.transform(MERCATOR, true);
        mercatorEnvelope.translate(mercatorEnvelope.getWidth() / 2.0, 0.0);
        Geometry g1 = new WKTReader().read("POLYGON((150 40, 150 -90, 190 -90, 190 40, 175 40, 175 -87, 165 -87, 165 40, 150 40))");
        Geometry g2 = new WKTReader().read("POLYGON((-178 -90, -178 90, 178 90, 178 -90, -178 -90))");
        MultiPolygon collection = new MultiPolygon(new Polygon[]{(Polygon)g1, (Polygon)g2}, g1.getFactory());
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)mercatorEnvelope, (CoordinateReferenceSystem)WGS84, (boolean)true);
        Assert.assertTrue((boolean)handler.requiresProcessing((Geometry)collection));
        Geometry preProcessed = handler.preProcess((Geometry)collection);
        Assert.assertNotEquals((Object)collection, (Object)preProcessed);
        Assert.assertNotNull((Object)preProcessed);
        Assert.assertTrue((boolean)(preProcessed instanceof GeometryCollection));
        int numGeometries = preProcessed.getNumGeometries();
        Assert.assertEquals((long)numGeometries, (long)3L);
        for (int i = 0; i < numGeometries; ++i) {
            Assert.assertTrue((boolean)(preProcessed.getGeometryN(i) instanceof Polygon));
        }
    }

    @Test
    public void testSkipEmptyGeometryCollections() throws Exception {
        ReferencedEnvelope world = new ReferencedEnvelope(160.0, 180.0, -40.0, 40.0, WGS84);
        ReferencedEnvelope mercatorEnvelope = world.transform(MERCATOR, true);
        mercatorEnvelope.translate(mercatorEnvelope.getWidth() / 2.0, 0.0);
        Geometry g1 = new WKTReader().read("POLYGON((150 40, 150 -90, 190 -90, 190 40, 175 40, 175 -87, 165 -87, 165 40, 150 40))");
        GeometryCollection collection = new GeometryCollection(null, g1.getFactory());
        ProjectionHandler handler = new ProjectionHandler(WGS84, new Envelope(-0.5, 2.0, -0.5, 2.0), mercatorEnvelope);
        Assert.assertTrue((boolean)handler.requiresProcessing((Geometry)collection));
        Geometry preProcessed = handler.preProcess((Geometry)collection);
        Assert.assertNotEquals((Object)collection, (Object)preProcessed);
        Assert.assertNull((Object)preProcessed);
    }

    @Test
    public void testQueryEnvelopesNonWrappingWGS84() throws Exception {
        ReferencedEnvelope request = new ReferencedEnvelope(170.0, 190.0, -40.0, 40.0, WGS84);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)request, (CoordinateReferenceSystem)WGS84, (boolean)false);
        List envelopes = handler.getQueryEnvelopes();
        Assert.assertEquals((long)1L, (long)envelopes.size());
        Assert.assertEquals((Object)request, envelopes.get(0));
    }

    @Test
    public void testQueryEnvelopesNonWrapping3857() throws Exception {
        ReferencedEnvelope requestWgs84 = new ReferencedEnvelope(170.0, 190.0, -40.0, 40.0, WGS84);
        ReferencedEnvelope request = requestWgs84.transform(OSM, true);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)request, (CoordinateReferenceSystem)WGS84, (boolean)false);
        List envelopes = handler.getQueryEnvelopes();
        Assert.assertEquals((long)1L, (long)envelopes.size());
        this.assertEnvelopesEqual(requestWgs84, (ReferencedEnvelope)envelopes.get(0), 1.0E-5);
    }

    private void assertEnvelopesEqual(ReferencedEnvelope re1, ReferencedEnvelope re2, double eps) {
        Assert.assertEquals((Object)re1.getCoordinateReferenceSystem(), (Object)re2.getCoordinateReferenceSystem());
        Assert.assertEquals((double)re1.getMinX(), (double)re2.getMinX(), (double)eps);
        Assert.assertEquals((double)re1.getMinY(), (double)re2.getMinY(), (double)eps);
        Assert.assertEquals((double)re1.getMaxX(), (double)re2.getMaxX(), (double)eps);
        Assert.assertEquals((double)re1.getMaxY(), (double)re2.getMaxY(), (double)eps);
    }

    @Test
    public void testQueryEnvelopesWrappingWGS84() throws Exception {
        ReferencedEnvelope request = new ReferencedEnvelope(170.0, 190.0, -40.0, 40.0, WGS84);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)request, (CoordinateReferenceSystem)WGS84, (boolean)true);
        List envelopes = handler.getQueryEnvelopes();
        Assert.assertEquals((long)2L, (long)envelopes.size());
        Assert.assertTrue((boolean)envelopes.contains(new ReferencedEnvelope(170.0, 180.0, -40.0, 40.0, WGS84)));
        Assert.assertTrue((boolean)envelopes.contains(new ReferencedEnvelope(-180.0, -170.0, -40.0, 40.0, WGS84)));
    }

    @Test
    public void testQueryEnvelopesWrapping3857() throws Exception {
        ReferencedEnvelope requestWgs84 = new ReferencedEnvelope(170.0, 190.0, -40.0, 40.0, WGS84);
        ReferencedEnvelope request = requestWgs84.transform(OSM, true);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)request, (CoordinateReferenceSystem)WGS84, (boolean)true);
        List envelopes = handler.getQueryEnvelopes();
        Assert.assertEquals((long)2L, (long)envelopes.size());
        this.assertEnvelopesEqual(new ReferencedEnvelope(170.0, 180.0, -40.0, 40.0, WGS84), (ReferencedEnvelope)envelopes.get(0), 1.0E-5);
        this.assertEnvelopesEqual(new ReferencedEnvelope(-180.0, -170.0, -40.0, 40.0, WGS84), (ReferencedEnvelope)envelopes.get(1), 1.0E-5);
    }

    @Test
    public void testWorldMeridian() throws Exception {
        ReferencedEnvelope requestWgs84 = new ReferencedEnvelope(-180.0, 180.0, -85.0, 85.0, WGS84);
        ReferencedEnvelope requestWebMercator = requestWgs84.transform(OSM, true);
        Geometry g = new WKTReader().read("LINESTRING(0 -90, 0 90)");
        Geometry expected = new WKTReader().read("LINESTRING(0 -85, 0 85)");
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)requestWebMercator, (CoordinateReferenceSystem)WGS84, (boolean)true);
        Assert.assertTrue((boolean)handler.requiresProcessing(g));
        Geometry preProcessed = handler.preProcess(g);
        Assert.assertEquals((Object)expected, (Object)preProcessed);
        Geometry postProcessed = handler.postProcess(CRS.findMathTransform((CoordinateReferenceSystem)WGS84, (CoordinateReferenceSystem)WGS84), expected);
        Assert.assertEquals((Object)expected, (Object)postProcessed);
    }

    @Test
    public void testWrapPDCMercator() throws Exception {
        CoordinateReferenceSystem pdc = CRS.decode((String)"EPSG:3832", (boolean)true);
        ReferencedEnvelope world = new ReferencedEnvelope(-2.0E7, 2.0E7, -2.0E7, 2.0E7, pdc);
        Geometry g = new WKTReader().read("MULTIPOLYGON(((-73 60, -73 83, -11 83, -11 60, -73 60)),((-10 60, -10 61, -11 61, -11 60, -10 60)))");
        Geometry original = g.copy();
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)world, (CoordinateReferenceSystem)WGS84, (boolean)true);
        Assert.assertTrue((boolean)handler.requiresProcessing(g));
        Geometry preProcessed = handler.preProcess(g);
        Assert.assertEquals((Object)original, (Object)preProcessed);
        MathTransform mt = CRS.findMathTransform((CoordinateReferenceSystem)WGS84, (CoordinateReferenceSystem)pdc, (boolean)true);
        Geometry transformed = JTS.transform((Geometry)g, (MathTransform)mt);
        Geometry postProcessed = handler.postProcess(mt, transformed);
        Assert.assertEquals((long)3L, (long)postProcessed.getNumGeometries());
        postProcessed.apply(geom -> {
            if (geom != postProcessed && geom.getEnvelopeInternal().getWidth() > 4.0E7) {
                Assert.fail((String)"The geometry did not get rewrapped properly");
            }
        });
    }

    @Test
    public void testReprojectBackwardsTo900913() throws Exception {
        String wkt = "PROJCS[\"WGS84 / Google Mercator\", GEOGCS[\"WGS 84\", DATUM[\"World Geodetic System 1984\", SPHEROID[\"WGS 84\", 6378137.0, 298.257223563, AUTHORITY[\"EPSG\",\"7030\"]], AUTHORITY[\"EPSG\",\"6326\"]], PRIMEM[\"Greenwich\", 0.0, AUTHORITY[\"EPSG\",\"8901\"]], UNIT[\"degree\", 0.017453292519943295], AUTHORITY[\"EPSG\",\"4326\"]], PROJECTION[\"Mercator (1SP)\", AUTHORITY[\"EPSG\",\"9804\"]], PARAMETER[\"semi_major\", 6378137.0], PARAMETER[\"semi_minor\", 6378137.0], PARAMETER[\"latitude_of_origin\", 0.0], PARAMETER[\"central_meridian\", 0.0], PARAMETER[\"scale_factor\", 1.0], PARAMETER[\"false_easting\", 0.0], PARAMETER[\"false_northing\", 0.0], UNIT[\"m\", 1.0],  AUTHORITY[\"EPSG\",\"900913\"]]";
        CoordinateReferenceSystem epsg900913 = CRS.parseWKT((String)wkt);
        ReferencedEnvelope renderingArea = new ReferencedEnvelope(-180.0, 0.0, 0.0, 90.0, (CoordinateReferenceSystem)DefaultGeographicCRS.WGS84);
        ProjectionHandler ph = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)renderingArea, (CoordinateReferenceSystem)epsg900913, (boolean)true);
        List queryEnvelopes = ph.getQueryEnvelopes();
        Assert.assertEquals((long)1L, (long)queryEnvelopes.size());
        ReferencedEnvelope qa = (ReferencedEnvelope)queryEnvelopes.get(0);
        Assert.assertEquals((double)-2.0037508342789244E7, (double)qa.getMinX(), (double)1.0);
        Assert.assertEquals((double)0.0, (double)qa.getMaxX(), (double)1.0);
        Assert.assertEquals((double)0.0, (double)qa.getMinY(), (double)1.0);
        Assert.assertEquals((double)2.8066337200331915E7, (double)qa.getMaxY(), (double)1.0);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testQueryEnvelopeOnInvalidArea() throws Exception {
        MapProjection.SKIP_SANITY_CHECKS = false;
        try {
            ReferencedEnvelope re = new ReferencedEnvelope(-130.0, -120.0, -40.0, 30.0, (CoordinateReferenceSystem)DefaultGeographicCRS.WGS84);
            ProjectionHandler ph = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)re, (CoordinateReferenceSystem)CRS.decode((String)"EPSG:3003", (boolean)true), (boolean)true);
            List queryEnvelopes = ph.getQueryEnvelopes();
            Assert.assertEquals((long)0L, (long)queryEnvelopes.size());
        }
        finally {
            MapProjection.SKIP_SANITY_CHECKS = true;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testQueryEnvelopeOnInvalidArea2() throws Exception {
        MapProjection.SKIP_SANITY_CHECKS = false;
        try {
            ReferencedEnvelope re = new ReferencedEnvelope(-130.0, -120.0, -40.0, 30.0, (CoordinateReferenceSystem)DefaultGeographicCRS.WGS84);
            ReferencedEnvelope re3857 = re.transform(CRS.decode((String)"EPSG:3857", (boolean)true), true);
            ProjectionHandler ph = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)re3857, (CoordinateReferenceSystem)CRS.decode((String)"EPSG:3003", (boolean)true), (boolean)true);
            List queryEnvelopes = ph.getQueryEnvelopes();
            Assert.assertEquals((long)0L, (long)queryEnvelopes.size());
        }
        finally {
            MapProjection.SKIP_SANITY_CHECKS = true;
        }
    }

    @Test
    public void testCutGeometryTouchingValidArea() throws Exception {
        String wkb = "AAAAAAMAAAACAAAAJsBX4dme0ZBrwBBHY3zBphTAV9wm/XQjVMAN5aZ25n+AwFfRnB4QLcDACkbY\nu6mfgMBX0IDk68FEwAn9i4xuQADAV9LyxRH1/MAKtRmwO4KAwFfFCYjFNfzABzwNbZWeAMBXro+u\nQ3TIwAMozHt2zADAV5nQ8kYDGMAABfJy8QWAwFdz2+lf1Xy/9dkUVMuzAMBXK24mVt9kv9u1veRJ\nCADAVxApDvwLFD0wAAAAAAAAwFlKt6hdgbi+jC+LAAAAAMBZS9W86QecwAP6Pg0jSwDAWOUBPzQM\nOMAD1MCXZ+6AwFiD2a7LN6jABO4o20CCgMBYcM4alj+AwAVxI3rh+4DAWGFB3LdZJMAGMd3VXb+A\nwFhJc+py+vDACDw/gWQHAMBYO+vyzRGUwAnbp1i4wgDAWDEaWTENgMALgYeOOnuAwFgpGBGENtjA\nDRepG9PXAMBYIP4dLlt8wA+klG+5IADAWBwX+BP4WMARKUKeEtiAwFgXoZhXNYDAExH5Zal0QMBY\nGEP/ANogwBbbF0xESIDAWBN/U/McQMAYXuP5jyKAwFgKqiyQ+gTAGix578FUgMBYA1s+gKpwwBsT\no7wYhYDAV/qZDD5aYMAbwu+JuyoAwFfqj6KbhxTAHIrvGplNAMBX4hOw7Jx0wBy+1+x/XkDAV9vV\nZESR7MAca5H4ZYHAwFfWkK/y7sDAG3TJ1EFXgMBX1+eqod08wBrqABEuC8DAV+jS2XIy3MAYBQgf\nRHnAwFfslYpsZkDAFphJ8XbhQMBX6mVp0arUwBOSV/ye1QDAV+HZntGQa8AQR2N8waYUAAAABMBX\n4dme0ZBrwBBHY3zBphTAV+GPbx9YxMAQMbM4fB1AwFfivDEmxxzAEIAAI+c6wMBX4dme0ZBrwBBH\nY3zBphQ=";
        Geometry geometry = new WKBReader().read(Base64.decode((String)wkb));
        CoordinateReferenceSystem lambertPolar = this.getLambertPolar();
        ReferencedEnvelope renderingEnvelope = new ReferencedEnvelope(-1.4542204652543461E7, 1.5480411404320458E7, -1.870549711355389E7, 1.1278026995319324E7, lambertPolar);
        ReferencedEnvelope validAreaBounds = new ReferencedEnvelope(-180.0, 180.0, 0.0, 90.0, (CoordinateReferenceSystem)DefaultGeographicCRS.WGS84);
        ProjectionHandler ph = new ProjectionHandler((CoordinateReferenceSystem)DefaultGeographicCRS.WGS84, (Envelope)validAreaBounds, renderingEnvelope);
        Assert.assertNull((Object)ph.preProcess(geometry));
    }

    @Test
    public void testCutGeometryCrossingValidArea() throws Exception {
        String wkb = "AAAAAAMAAAABAAAAIMBX4dme0ZBrwBBHY3zBphTAV9GcHhAtwMAKRti7qZ+AwFfQgOTrwUTACf2L\njG5AAMBX0vLFEfX8wAq1GbA7goDAV8UJiMU1/MAHPA1tlZ4AwFeuj65DdMjAAyjMe3bMAMBXc9vp\nX9V8v/XZFFTLswDAVytuJlbfZL/btb3kSQgAwFcQKQ78CxQ9MAAAAAAAAMBZSreoXYG4vowviwAA\nAADAWUvVvOkHnMAD+j4NI0sAwFjlAT80DDjAA9TAl2fugMBYg9muyzeowATuKNtAgoDAWGFB3LdZ\nJMAGMd3VXb+AwFhJc+py+vDACDw/gWQHAMBYMRpZMQ2AwAuBh446e4DAWCkYEYQ22MANF6kb09cA\nwFgg/h0uW3zAD6SUb7kgAMBYF6GYVzWAwBMR+WWpdEDAWBhD/wDaIMAW2xdMREiAwFgKqiyQ+gTA\nGix578FUgMBYA1s+gKpwwBsTo7wYhYDAV/qZDD5aYMAbwu+JuyoAwFfqj6KbhxTAHIrvGplNAMBX\n4hOw7Jx0wBy+1+x/XkDAV9vVZESR7MAca5H4ZYHAwFfWkK/y7sDAG3TJ1EFXgMBX1+eqod08wBrq\nABEuC8DAV+jS2XIy3MAYBQgfRHnAwFfslYpsZkDAFphJ8XbhQMBX6mVp0arUwBOSV/ye1QDAV+HZ\nntGQa8AQR2N8waYU";
        Geometry geometry = new WKBReader().read(Base64.decode((String)wkb));
        CoordinateReferenceSystem lambertPolar = this.getLambertPolar();
        ReferencedEnvelope renderingEnvelope = new ReferencedEnvelope(-1.4542204652543461E7, 1.5480411404320458E7, -1.870549711355389E7, 1.1278026995319324E7, lambertPolar);
        ReferencedEnvelope validAreaBounds = new ReferencedEnvelope(-180.0, 180.0, 0.0, 90.0, (CoordinateReferenceSystem)DefaultGeographicCRS.WGS84);
        ProjectionHandler ph = new ProjectionHandler((CoordinateReferenceSystem)DefaultGeographicCRS.WGS84, (Envelope)validAreaBounds, renderingEnvelope);
        Geometry preProcessed = ph.preProcess(geometry);
        Assert.assertNull((Object)preProcessed);
    }

    private CoordinateReferenceSystem getLambertPolar() throws FactoryException {
        String crsWKT = "PROJCS[\"North_Pole_Lambert_Azimuthal_Equal_Area\",GEOGCS[\"GCS_WGS_1984\",DATUM[\"D_WGS_1984\",SPHEROID[\"WGS_1984\",6378137,298.257223563]],PRIMEM[\"Greenwich\",0],UNIT[\"Degree\",0.017453292519943295]],PROJECTION[\"Lambert_Azimuthal_Equal_Area\"],PARAMETER[\"False_Easting\",0],PARAMETER[\"False_Northing\",0],PARAMETER[\"Central_Meridian\",0],PARAMETER[\"Latitude_Of_Origin\",90],UNIT[\"Meter\",1]]";
        return CRS.parseWKT((String)crsWKT);
    }

    @Test
    public void testUTMDatelineWrapping() throws Exception {
        CoordinateReferenceSystem crs = CRS.decode((String)"EPSG:32601", (boolean)true);
        ReferencedEnvelope re = new ReferencedEnvelope(300000.0, 409800.0, 5890200.0, 6000000.0, crs);
        MathTransform mt = CRS.findMathTransform((CoordinateReferenceSystem)crs, (CoordinateReferenceSystem)WGS84);
        Polygon geom = JTS.toGeometry((ReferencedEnvelope)re);
        ReferencedEnvelope targetReferenceEnvelope = new ReferencedEnvelope(-180.0, 180.0, -90.0, 90.0, WGS84);
        ProjectionHandler ph = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)targetReferenceEnvelope, (CoordinateReferenceSystem)crs, (boolean)true);
        Geometry preProcessed = ph.preProcess((Geometry)geom);
        Geometry transformed = JTS.transform((Geometry)preProcessed, (MathTransform)mt);
        Geometry postProcessed = ph.postProcess(mt.inverse(), transformed);
        Envelope ppEnvelope = postProcessed.getGeometryN(0).getEnvelopeInternal();
        Assert.assertTrue((boolean)ppEnvelope.contains(180.0, 54.0));
        Assert.assertEquals((double)1.7, (double)ppEnvelope.getWidth(), (double)0.1);
    }

    @Test
    public void testMercatorBug() throws NoSuchAuthorityCodeException, FactoryException, TransformException {
        CoordinateReferenceSystem sourceCrs = CRS.decode((String)"EPSG:3857", (boolean)true);
        CoordinateReferenceSystem targetCrs = CRS.decode((String)"EPSG:31370", (boolean)true);
        ReferencedEnvelope sourceEnv = new ReferencedEnvelope(381033.2707188717, 381046.4083331082, 6583847.177786637, 6583860.315400874, sourceCrs);
        ProjectionHandler projectionHandler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)sourceEnv, (CoordinateReferenceSystem)targetCrs, (boolean)true);
        ReferencedEnvelope targetEnv = (ReferencedEnvelope)projectionHandler.getQueryEnvelopes().get(0);
        Assert.assertEquals((double)83304.59570855058, (double)targetEnv.getMinX(), (double)0.1);
        Assert.assertEquals((double)83313.02253560493, (double)targetEnv.getMaxX(), (double)0.1);
        Assert.assertEquals((double)164573.9584101988, (double)targetEnv.getMinY(), (double)0.1);
        Assert.assertEquals((double)164582.36316849105, (double)targetEnv.getMaxY(), (double)0.1);
    }

    @Test
    public void testWGS84BackToWebMercator() throws Exception {
        ReferencedEnvelope renderingEnvelope = new ReferencedEnvelope(135.0, 180.0, -90.0, -45.0, WGS84);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)renderingEnvelope, (CoordinateReferenceSystem)OSM, (boolean)true);
        Assert.assertNotNull((Object)handler);
        List queryEnvelopes = handler.getQueryEnvelopes();
        Assert.assertEquals((long)1L, (long)queryEnvelopes.size());
        ReferencedEnvelope expected = new ReferencedEnvelope(1.5028131257091932E7, 2.0037508342789244E7, -3.2487565023661762E7, -5621521.486192067, OSM);
        this.assertEnvelopesEqual(expected, (ReferencedEnvelope)queryEnvelopes.get(0), 1.0E-5);
    }

    @Test
    public void testE50LatLonBackToWebMercator() throws Exception {
        ReferencedEnvelope renderingEnvelope = new ReferencedEnvelope(-80.0, -45.0, 135.0, 180.0, ED50_LATLON);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)renderingEnvelope, (CoordinateReferenceSystem)OSM, (boolean)true);
        Assert.assertNotNull((Object)handler);
        List queryEnvelopes = handler.getQueryEnvelopes();
        Assert.assertEquals((long)1L, (long)queryEnvelopes.size());
        ReferencedEnvelope expected = new ReferencedEnvelope(-2.003748375258002E7, 1.958231203373337E7, -1.5538175797794182E7, -5621345.809658899, OSM);
        this.assertEnvelopesEqual(expected, (ReferencedEnvelope)queryEnvelopes.get(0), 1.0E-5);
    }

    @Test
    public void testE50BackToWebMercator() throws Exception {
        ReferencedEnvelope renderingEnvelope = new ReferencedEnvelope(135.0, 180.0, -80.0, -45.0, ED50);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)renderingEnvelope, (CoordinateReferenceSystem)OSM, (boolean)true);
        Assert.assertNotNull((Object)handler);
        List queryEnvelopes = handler.getQueryEnvelopes();
        Assert.assertEquals((long)1L, (long)queryEnvelopes.size());
        ReferencedEnvelope expected = new ReferencedEnvelope(-2.003748375258002E7, 1.958231203373337E7, -1.5538175797794182E7, -5621345.809658899, OSM);
        this.assertEnvelopesEqual(expected, (ReferencedEnvelope)queryEnvelopes.get(0), 1.0E-5);
    }

    @Test
    public void testOutsideValidArea() throws Exception {
        Geometry g = new WKTReader().read("POLYGON((0 87, 10 87, 10 89, 0 89, 0 87))");
        Geometry original = g.copy();
        ReferencedEnvelope re = new ReferencedEnvelope(-2.0E7, 2.0E7, -2.0E7, 2.0E7, OSM);
        ProjectionHandler ph = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)re, (CoordinateReferenceSystem)WGS84, (boolean)true);
        Geometry preProcessed = ph.preProcess(original);
        Assert.assertNull((Object)preProcessed);
    }

    @Test
    public void testLargeObject() throws Exception {
        Geometry g = new WKTReader().read("POLYGON((-96 -2, -96 67, 133 67, 133 -2, -96 -2))");
        Geometry original = g.copy();
        ReferencedEnvelope re = new ReferencedEnvelope(-4.0E7, 4.0E7, -2.0E7, 2.0E7, OSM);
        ProjectionHandler ph = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)re, (CoordinateReferenceSystem)WGS84, (boolean)true);
        Geometry preProcessed = ph.preProcess(original);
        MathTransform mt = CRS.findMathTransform((CoordinateReferenceSystem)WGS84, (CoordinateReferenceSystem)OSM, (boolean)true);
        Geometry transformed = JTS.transform((Geometry)preProcessed, (MathTransform)mt);
        Geometry postProcessed = ph.postProcess(mt.inverse(), transformed);
        MatcherAssert.assertThat((Object)postProcessed, (Matcher)CoreMatchers.instanceOf(MultiPolygon.class));
        Assert.assertEquals((long)3L, (long)postProcessed.getNumGeometries());
        Assert.assertEquals((double)2.54E7, (double)postProcessed.getGeometryN(0).getEnvelopeInternal().getWidth(), (double)100000.0);
        Assert.assertEquals((double)2.54E7, (double)postProcessed.getGeometryN(1).getEnvelopeInternal().getWidth(), (double)100000.0);
        Assert.assertEquals((double)2.54E7, (double)postProcessed.getGeometryN(2).getEnvelopeInternal().getWidth(), (double)100000.0);
    }

    @Test
    public void testLargeObjectSourceInFeet() throws Exception {
        String worldMercatorFeet = "PROJCS[\"World_Mercator\",\n    GEOGCS[\"GCS_WGS_1984\",\n        DATUM[\"WGS_1984\",\n            SPHEROID[\"WGS_1984\",6378137,298.257223563]],\n        PRIMEM[\"Greenwich\",0],\n        UNIT[\"Degree\",0.017453292519943295]],\n    PROJECTION[\"Mercator_1SP\"],\n    PARAMETER[\"False_Easting\",0],\n    PARAMETER[\"False_Northing\",0],\n    PARAMETER[\"Central_Meridian\",0],\n    UNIT[\"Foot_US\",0.3048006096012192],\n    AUTHORITY[\"EPSG\",\"54004\"]]";
        CoordinateReferenceSystem WOLD_MERCATOR_FEET = CRS.parseWKT((String)worldMercatorFeet);
        Geometry g = new WKTReader().read("POLYGON ((-35061186 -725700, -35061186 33191143, 48574352 33191143, 48574352 -725700, -35061186 -725700))");
        Geometry original = g.copy();
        ReferencedEnvelope re = new ReferencedEnvelope(-540.0, 540.0, -90.0, 90.0, WGS84);
        ProjectionHandler ph = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)re, (CoordinateReferenceSystem)WOLD_MERCATOR_FEET, (boolean)true);
        Geometry preProcessed = ph.preProcess(original);
        MathTransform mt = CRS.findMathTransform((CoordinateReferenceSystem)WOLD_MERCATOR_FEET, (CoordinateReferenceSystem)WGS84, (boolean)true);
        Geometry transformed = JTS.transform((Geometry)preProcessed, (MathTransform)mt);
        Geometry postProcessed = ph.postProcess(mt.inverse(), transformed);
        MatcherAssert.assertThat((Object)postProcessed, (Matcher)CoreMatchers.instanceOf(MultiPolygon.class));
        Assert.assertEquals((long)3L, (long)postProcessed.getNumGeometries());
        Assert.assertEquals((double)228.0, (double)postProcessed.getGeometryN(0).getEnvelopeInternal().getWidth(), (double)1.0);
        Assert.assertEquals((double)228.0, (double)postProcessed.getGeometryN(1).getEnvelopeInternal().getWidth(), (double)1.0);
        Assert.assertEquals((double)228.0, (double)postProcessed.getGeometryN(2).getEnvelopeInternal().getWidth(), (double)1.0);
    }

    @Test
    public void testAzEqBeyondLimitsCentered() throws Exception {
        CoordinateReferenceSystem crs = CRS.decode((String)"AUTO:97003,9001,0,0", (boolean)true);
        double beyond = 2.7E7;
        ReferencedEnvelope re = new ReferencedEnvelope(-beyond, beyond, -beyond, beyond, crs);
        ProjectionHandler ph = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)re, (CoordinateReferenceSystem)DefaultGeographicCRS.WGS84, (boolean)false);
        Assert.assertNotNull((Object)ph);
        List envelopes = ph.getQueryEnvelopes();
        Assert.assertEquals((long)1L, (long)envelopes.size());
        ReferencedEnvelope qe = (ReferencedEnvelope)envelopes.get(0);
        Assert.assertEquals((double)-180.0, (double)qe.getMinX(), (double)0.001);
        Assert.assertEquals((double)-90.0, (double)qe.getMinY(), (double)0.001);
        Assert.assertEquals((double)180.0, (double)qe.getMaxX(), (double)0.001);
        Assert.assertEquals((double)90.0, (double)qe.getMaxY(), (double)0.001);
    }

    @Test
    public void testAzEqBeyondLimits150_60() throws Exception {
        CoordinateReferenceSystem crs = CRS.decode((String)"AUTO:97003,9001,150,60", (boolean)true);
        double beyond = 2.7E7;
        ReferencedEnvelope re = new ReferencedEnvelope(-beyond, beyond, -beyond, beyond, crs);
        ProjectionHandler ph = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)re, (CoordinateReferenceSystem)DefaultGeographicCRS.WGS84, (boolean)false);
        Assert.assertNotNull((Object)ph);
        List envelopes = ph.getQueryEnvelopes();
        Assert.assertEquals((long)1L, (long)envelopes.size());
        ReferencedEnvelope qe = (ReferencedEnvelope)envelopes.get(0);
        Assert.assertEquals((double)-180.0, (double)qe.getMinX(), (double)0.001);
        Assert.assertEquals((double)-90.0, (double)qe.getMinY(), (double)0.001);
        Assert.assertEquals((double)180.0, (double)qe.getMaxX(), (double)0.001);
        Assert.assertEquals((double)90.0, (double)qe.getMaxY(), (double)0.001);
    }

    @Test
    public void testAzEqBeyondLimitsHalfWorld() throws Exception {
        CoordinateReferenceSystem crs = CRS.decode((String)"AUTO:97003,9001,0,0", (boolean)true);
        double beyond = 2.7E7;
        ReferencedEnvelope re = new ReferencedEnvelope(0.0, beyond, -beyond, beyond, crs);
        ProjectionHandler ph = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)re, (CoordinateReferenceSystem)DefaultGeographicCRS.WGS84, (boolean)false);
        Assert.assertNotNull((Object)ph);
        List envelopes = ph.getQueryEnvelopes();
        Assert.assertEquals((long)1L, (long)envelopes.size());
        ReferencedEnvelope qe = (ReferencedEnvelope)envelopes.get(0);
        Assert.assertEquals((double)0.0, (double)qe.getMinX(), (double)0.001);
        Assert.assertEquals((double)-90.0, (double)qe.getMinY(), (double)0.001);
        Assert.assertEquals((double)180.0, (double)qe.getMaxX(), (double)0.001);
        Assert.assertEquals((double)90.0, (double)qe.getMaxY(), (double)0.001);
    }

    @Test
    public void testAzEqBeyondLimitsQuarterWorld() throws Exception {
        CoordinateReferenceSystem crs = CRS.decode((String)"AUTO:97003,9001,0,0", (boolean)true);
        double beyond = 2.7E7;
        ReferencedEnvelope re = new ReferencedEnvelope(0.0, beyond, 0.0, beyond, crs);
        ProjectionHandler ph = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)re, (CoordinateReferenceSystem)DefaultGeographicCRS.WGS84, (boolean)false);
        Assert.assertNotNull((Object)ph);
        List envelopes = ph.getQueryEnvelopes();
        Assert.assertEquals((long)1L, (long)envelopes.size());
        ReferencedEnvelope qe = (ReferencedEnvelope)envelopes.get(0);
        Assert.assertEquals((double)0.0, (double)qe.getMinX(), (double)0.2);
        Assert.assertEquals((double)0.0, (double)qe.getMinY(), (double)0.2);
        Assert.assertEquals((double)180.0, (double)qe.getMaxX(), (double)0.001);
        Assert.assertEquals((double)90.0, (double)qe.getMaxY(), (double)0.001);
    }

    @Test
    public void testQueryEnvelopeAcrossDateLine() throws Exception {
        double minX = 140.0;
        double maxX = 200.0;
        ReferencedEnvelope re = new ReferencedEnvelope(140.0, 200.0, -40.0, 30.0, WGS84);
        ProjectionHandler ph = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)re, (CoordinateReferenceSystem)WGS84, (boolean)true);
        List queryEnvelopes = ph.getQueryEnvelopes();
        Assert.assertEquals((long)2L, (long)queryEnvelopes.size());
        ReferencedEnvelope qe = (ReferencedEnvelope)queryEnvelopes.get(1);
        double wrappedMinX = 140.0;
        double wrappedMaxX = 200.0;
        Assert.assertEquals((double)-180.0, (double)qe.getMinX(), (double)0.001);
        Assert.assertEquals((double)-160.0, (double)qe.getMaxX(), (double)0.001);
        qe = (ReferencedEnvelope)queryEnvelopes.get(0);
        Assert.assertEquals((double)140.0, (double)qe.getMinX(), (double)0.001);
        Assert.assertEquals((double)180.0, (double)qe.getMaxX(), (double)0.001);
    }

    @Test
    public void testQueryEnvelopeFarAway() throws Exception {
        double minX = 2170.0;
        double maxX = 2220.0;
        ReferencedEnvelope re = new ReferencedEnvelope(2170.0, 2220.0, -40.0, 30.0, WGS84);
        ProjectionHandler ph = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)re, (CoordinateReferenceSystem)WGS84, (boolean)true);
        List queryEnvelopes = ph.getQueryEnvelopes();
        ReferencedEnvelope qe = (ReferencedEnvelope)queryEnvelopes.get(queryEnvelopes.size() - 1);
        double wrappedMinX = 10.0;
        double wrappedMaxX = 60.0;
        Assert.assertEquals((double)10.0, (double)qe.getMinX(), (double)0.001);
        Assert.assertEquals((double)60.0, (double)qe.getMaxX(), (double)0.001);
    }

    @Test
    public void testQueryEnvelopeAcrossDateLineFarAway() throws Exception {
        double positiveMinX = 2170.0;
        double positiveMaxX = 2380.0;
        double negativeMinX = -2380.0;
        double negativeMaxX = -2170.0;
        double[] negatives = new double[]{-2380.0, -2170.0};
        double[] positives = new double[]{2170.0, 2380.0};
        double[][] sets = new double[][]{negatives, positives};
        boolean negative = true;
        for (double[] set : sets) {
            double minX = set[0];
            double maxX = set[1];
            ReferencedEnvelope re = new ReferencedEnvelope(minX, maxX, -40.0, 30.0, WGS84);
            ProjectionHandler ph = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)re, (CoordinateReferenceSystem)WGS84, (boolean)true);
            List queryEnvelopes = ph.getQueryEnvelopes();
            Assert.assertEquals((long)3L, (long)queryEnvelopes.size());
            ReferencedEnvelope qe = (ReferencedEnvelope)queryEnvelopes.get(1);
            double wrappedMinX = minX % 360.0;
            double wrappedMaxX = maxX % 360.0;
            Assert.assertEquals((double)-180.0, (double)qe.getMinX(), (double)0.001);
            Assert.assertEquals((double)(wrappedMaxX - (double)(negative ? 0 : 360)), (double)qe.getMaxX(), (double)0.001);
            qe = (ReferencedEnvelope)queryEnvelopes.get(2);
            Assert.assertEquals((double)(wrappedMinX + (double)(negative ? 360 : 0)), (double)qe.getMinX(), (double)0.001);
            Assert.assertEquals((double)180.0, (double)qe.getMaxX(), (double)0.001);
            negative = false;
        }
    }

    @Test
    public void testQueryEnvelopeOnExtentGreaterThanWholeWorld() throws Exception {
        ReferencedEnvelope re = new ReferencedEnvelope(350.0, 1000.0, -40.0, 30.0, WGS84);
        ProjectionHandler ph = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)re, (CoordinateReferenceSystem)OSM, (boolean)true);
        List queryEnvelopes = ph.getQueryEnvelopes();
        ReferencedEnvelope qe = (ReferencedEnvelope)queryEnvelopes.get(queryEnvelopes.size() - 1);
        MathTransform transform = CRS.findMathTransform((CoordinateReferenceSystem)WGS84, (CoordinateReferenceSystem)OSM);
        Coordinate minX = JTS.transform((Coordinate)new Coordinate(-180.0, -85.0), null, (MathTransform)transform);
        Coordinate maxX = JTS.transform((Coordinate)new Coordinate(180.0, -85.0), null, (MathTransform)transform);
        Assert.assertEquals((double)minX.x, (double)qe.getMinX(), (double)0.001);
        Assert.assertEquals((double)maxX.x, (double)qe.getMaxX(), (double)0.001);
        ph = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)re, (CoordinateReferenceSystem)WGS84, (boolean)true);
        queryEnvelopes = ph.getQueryEnvelopes();
        qe = (ReferencedEnvelope)queryEnvelopes.get(queryEnvelopes.size() - 1);
        Assert.assertEquals((double)-180.0, (double)qe.getMinX(), (double)0.001);
        Assert.assertEquals((double)180.0, (double)qe.getMaxX(), (double)0.001);
    }

    @Test
    public void testCutGeometryHomolosine() throws Exception {
        String wkt = "PROJCS[\"Homolosine\",GEOGCS[\"WGS 84\",DATUM[\"WGS_1984\",SPHEROID[\"WGS 84\",6378137,298.257223563 ] ], PRIMEM[\"Greenwich\",0.0], UNIT[\"degree\",0.01745329251994328 ]],PROJECTION[\"Goode_Homolosine\"],UNIT[\"m\",1.0] ]";
        CoordinateReferenceSystem homolosine = CRS.parseWKT((String)wkt);
        Envelope worldEnvelope = new Envelope(-180.0, 180.0, -90.0, 90.0);
        ReferencedEnvelope worldWGS84 = new ReferencedEnvelope(worldEnvelope, WGS84);
        ReferencedEnvelope worldHomolosine = worldWGS84.transform(homolosine, true);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)worldHomolosine, (CoordinateReferenceSystem)WGS84, (boolean)true);
        Polygon geometry = JTS.toGeometry((Envelope)worldEnvelope);
        Geometry preProcessed = handler.preProcess((Geometry)geometry);
        Assert.assertEquals((Object)worldEnvelope, (Object)preProcessed.getEnvelopeInternal());
        GeometryFactory gf = geometry.getFactory();
        Assert.assertFalse((boolean)preProcessed.intersects((Geometry)this.lineString(gf, new double[]{-40.0, 0.1, -40.0, 90.0})));
        Assert.assertFalse((boolean)preProcessed.intersects((Geometry)this.lineString(gf, new double[]{-100.0, -0.1, -100.0, -90.0})));
        Assert.assertFalse((boolean)preProcessed.intersects((Geometry)this.lineString(gf, new double[]{-20.0, -0.1, -20.0, -90.0})));
        Assert.assertFalse((boolean)preProcessed.intersects((Geometry)this.lineString(gf, new double[]{80.0, -0.1, 80.0, -90.0})));
    }

    public LineString lineString(GeometryFactory gf, double[] coords) {
        return gf.createLineString((CoordinateSequence)new LiteCoordinateSequence(coords));
    }

    @Test
    public void testRotatedPolarSource() throws Exception {
        CoordinateReferenceSystem rotatedPolar = CRS.parseWKT((String)"FITTED_CS[\"rotated_latitude_longitude\", INVERSE_MT[PARAM_MT[\"Rotated_Pole\",  PARAMETER[\"semi_major\", 6371229.0],  PARAMETER[\"semi_minor\", 6371229.0],  PARAMETER[\"central_meridian\", -106.0],  PARAMETER[\"latitude_of_origin\", 54.0],  PARAMETER[\"scale_factor\", 1.0],  PARAMETER[\"false_easting\", 0.0],  PARAMETER[\"false_northing\", 0.0]]],  GEOGCS[\"unknown\", DATUM[\"unknown\",  SPHEROID[\"unknown\", 6371229.0, 0.0]],  PRIMEM[\"Greenwich\", 0.0],  UNIT[\"degree\", 0.017453292519943295],  AXIS[\"Geodetic longitude\", EAST],  AXIS[\"Geodetic latitude\", NORTH]]]");
        Envelope worldEnvelope = new Envelope(-180.0, 180.0, -90.0, 90.0);
        ReferencedEnvelope worldWGS84 = new ReferencedEnvelope(worldEnvelope, WGS84);
        ProjectionHandler handler = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)worldWGS84, (CoordinateReferenceSystem)rotatedPolar, (boolean)true);
        MatcherAssert.assertThat((Object)handler, (Matcher)Matchers.instanceOf(WrappingProjectionHandler.class));
    }

    @Test
    public void testAzEqFalseOrigins() throws Exception {
        String wkt = "PROJCS[\"equi7_antarctica\",GEOGCS[\"GCS_WGS_1984\",DATUM[\"D_WGS_1984\",SPHEROID[\"WGS_1984\",6378137.0,298.257223563]],PRIMEM[\"Greenwich\",0.0],UNIT[\"Degree\",0.0174532925199433]],PROJECTION[\"Azimuthal_Equidistant\"],PARAMETER[\"false_easting\",3714266.97719],PARAMETER[\"false_northing\",3402016.50625],PARAMETER[\"central_meridian\",0.0],PARAMETER[\"latitude_of_origin\",-90.0],UNIT[\"Meter\",1.0]]";
        CoordinateReferenceSystem crs = CRS.parseWKT((String)wkt);
        double beyond = 2.7E7;
        double cx = 3714266.97719;
        double cy = 3402016.50625;
        ReferencedEnvelope re = new ReferencedEnvelope(cx - beyond, cx + beyond, cy - beyond, cy + beyond, crs);
        ProjectionHandler ph = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)re, (CoordinateReferenceSystem)DefaultGeographicCRS.WGS84, (boolean)false);
        Assert.assertNotNull((Object)ph);
        List envelopes = ph.getQueryEnvelopes();
        Assert.assertEquals((long)1L, (long)envelopes.size());
        ReferencedEnvelope qe = (ReferencedEnvelope)envelopes.get(0);
        Assert.assertEquals((double)-180.0, (double)qe.getMinX(), (double)0.001);
        Assert.assertEquals((double)-90.0, (double)qe.getMinY(), (double)0.001);
        Assert.assertEquals((double)180.0, (double)qe.getMaxX(), (double)0.001);
        Assert.assertEquals((double)90.0, (double)qe.getMaxY(), (double)0.001);
    }

    @Test
    public void testAzEqPositiveLatOrigin() throws Exception {
        String wkt = "PROJCS[\"equi7_asia_nofalseXY\",GEOGCS[\"GCS_WGS_1984\",DATUM[\"D_WGS_1984\",SPHEROID[\"WGS_1984\",6378137.0,298.257223563]],PRIMEM[\"Greenwich\",0.0],UNIT[\"Degree\",0.0174532925199433]],PROJECTION[\"Azimuthal_Equidistant\"],PARAMETER[\"false_easting\",4340913.84808],PARAMETER[\"false_northing\",4812712.92347],PARAMETER[\"central_meridian\",94.0],PARAMETER[\"latitude_of_origin\",47.0],UNIT[\"Meter\",1.0]]";
        CoordinateReferenceSystem crs = CRS.parseWKT((String)wkt);
        ReferencedEnvelope re = new ReferencedEnvelope(-1.2E7, 1.2E7, -1.2E7, 1.2E7, crs);
        ProjectionHandler ph = ProjectionHandlerFinder.getHandler((ReferencedEnvelope)re, (CoordinateReferenceSystem)DefaultGeographicCRS.WGS84, (boolean)false);
        Assert.assertNotNull((Object)ph);
        List envelopes = ph.getQueryEnvelopes();
        Assert.assertEquals((long)1L, (long)envelopes.size());
        ReferencedEnvelope qe = (ReferencedEnvelope)envelopes.get(0);
        Assert.assertEquals((double)-180.0, (double)qe.getMinX(), (double)0.001);
        Assert.assertEquals((double)-90.0, (double)qe.getMinY(), (double)0.001);
        Assert.assertEquals((double)180.0, (double)qe.getMaxX(), (double)0.001);
        Assert.assertEquals((double)90.0, (double)qe.getMaxY(), (double)0.001);
    }

    static {
        gf = new GeometryFactory();
    }
}

