/*
 * Decompiled with CFR 0.152.
 */
package org.geotools.data.geoparquet;

import java.io.IOException;
import java.net.URI;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Function;
import java.util.function.UnaryOperator;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.geotools.api.data.Transaction;
import org.geotools.api.feature.simple.SimpleFeatureType;
import org.geotools.data.geoparquet.GeoParquetConfig;
import org.geotools.data.geoparquet.HivePartitionResolver;
import org.geotools.jdbc.JDBCDataStore;
import org.geotools.util.logging.Logging;

class GeoParquetViewManager {
    private static final Logger LOGGER = Logging.getLogger(GeoParquetViewManager.class);
    private final JDBCDataStore dataStore;
    private GeoParquetConfig config;
    private Map<String, Partition> partitionsByViewName = Map.of();

    public GeoParquetViewManager(JDBCDataStore dataStore) {
        this.dataStore = dataStore;
    }

    public GeoParquetConfig getConfig() {
        return this.config;
    }

    public void initialize(GeoParquetConfig config) throws IOException {
        URI targetUri = config.getTargetUri();
        Integer maxHiveDepth = config.getMaxHiveDepth();
        LOGGER.config("Resolving files for geoparquet uri " + targetUri);
        Map<String, Partition> partitions = this.loadPartitions(targetUri, maxHiveDepth);
        LOGGER.log(Level.CONFIG, () -> String.format("Found %,d partitions with %,d total files", partitions.keySet().size(), partitions.values().stream().map(Partition::getFiles).flatMap(Collection::stream).count()));
        this.dropViews();
        this.config = config;
        this.partitionsByViewName = partitions;
    }

    private Map<String, Partition> loadPartitions(URI targetUri, Integer maxHiveDepth) throws IOException {
        Map<String, List<String>> partitionFilesByUrl = this.findPartitions(targetUri, maxHiveDepth);
        return partitionFilesByUrl.entrySet().stream().map(e -> this.newPartition((String)e.getKey(), (List)e.getValue())).collect(Collectors.toMap(Partition::getViewName, Function.identity()));
    }

    private Partition newPartition(String uri, List<String> files) {
        String viewName = HivePartitionResolver.buildPartitionName(uri);
        return new Partition(uri, viewName, files);
    }

    private synchronized void dropViews() throws IOException {
        try (Connection c = this.getConnection();
             Statement st = c.createStatement();){
            for (Partition p : this.partitionsByViewName.values()) {
                String view = p.getViewName();
                String sql = String.format("DROP VIEW IF EXISTS \"%s\"", view);
                st.addBatch(sql);
                LOGGER.log(Level.CONFIG, () -> String.format("Dropping view %s for URI %s", view, p.getURI()));
            }
            st.executeBatch();
        }
        catch (SQLException e) {
            throw new IOException(e);
        }
    }

    public void createViewIfNotExists(String viewName) throws IOException {
        Partition partition = Objects.requireNonNull(this.partitionsByViewName.get(viewName));
        partition.ensureRegistered();
    }

    public SimpleFeatureType getViewFeatureType(SimpleFeatureType original, UnaryOperator<SimpleFeatureType> builder) {
        Partition partition = Objects.requireNonNull(this.partitionsByViewName.get(original.getTypeName()));
        return partition.getFeatureType(original, builder);
    }

    public List<String> getViewNames() {
        return this.partitionsByViewName.keySet().stream().sorted().collect(Collectors.toList());
    }

    public String getVieUri(String viewName) {
        return Objects.requireNonNull(this.partitionsByViewName.get(Objects.requireNonNull(viewName, "viewName")).getURI(), () -> String.format("No target URL exists for view %s", viewName));
    }

    private Map<String, List<String>> findPartitions(URI targetUri, Integer maxHiveDepth) throws IOException {
        Map<String, List<String>> partitionedFiles;
        try (Connection c = this.getConnection();){
            partitionedFiles = HivePartitionResolver.getHivePartitionedFiles(c, targetUri, maxHiveDepth);
        }
        catch (SQLException e) {
            throw new IOException(e);
        }
        return partitionedFiles;
    }

    public Connection getConnection() throws IOException {
        return this.dataStore.getConnection(Transaction.AUTO_COMMIT);
    }

    private class Partition {
        private final Lock lock = new ReentrantLock();
        private final AtomicBoolean registered = new AtomicBoolean();
        private final String uri;
        private final String viewName;
        private final List<String> files;
        private SimpleFeatureType viewType;

        public Partition(String uri, String viewName, List<String> files) {
            this.uri = Objects.requireNonNull(uri);
            this.viewName = Objects.requireNonNull(viewName);
            this.files = Objects.requireNonNull(files);
        }

        public String getViewName() {
            return this.viewName;
        }

        public String getURI() {
            return this.uri;
        }

        public List<String> getFiles() {
            return this.files;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public SimpleFeatureType getFeatureType(SimpleFeatureType original, UnaryOperator<SimpleFeatureType> builder) {
            SimpleFeatureType finalType = this.viewType;
            if (finalType == null) {
                this.lock.lock();
                try {
                    if (this.viewType == null) {
                        this.viewType = finalType = (SimpleFeatureType)builder.apply(original);
                    } else {
                        finalType = this.viewType;
                    }
                }
                finally {
                    this.lock.unlock();
                }
            }
            return finalType;
        }

        public void ensureRegistered() throws IOException {
            this.lock.lock();
            try {
                if (this.registered.compareAndSet(false, true)) {
                    String partitionUrl = this.getURI();
                    this.createView(this.viewName, partitionUrl);
                }
            }
            finally {
                this.lock.unlock();
            }
        }

        private void createView(String viewName, String partitionUrl) throws IOException {
            LOGGER.log(Level.INFO, () -> String.format("Creating view %s for URI %s", viewName, partitionUrl));
            String viewSql = this.createViewSql(viewName, partitionUrl);
            try (Connection c = GeoParquetViewManager.this.getConnection();
                 Statement st = c.createStatement();){
                st.execute(viewSql);
            }
            catch (SQLException e) {
                throw new IOException(e);
            }
            LOGGER.log(Level.INFO, () -> String.format("Created view %s for URI %s", viewName, partitionUrl));
        }

        private String createViewSql(String viewName, String partitionUrl) {
            return String.format("CREATE OR REPLACE VIEW \"%s\" AS SELECT * FROM read_parquet('%s', union_by_name = true)", viewName, partitionUrl);
        }
    }
}

