/*
 *    GeoTools - The Open Source Java GIS Toolkit
 *    http://geotools.org
 *
 *    (C) 2002-2025, Open Source Geospatial Foundation (OSGeo)
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation;
 *    version 2.1 of the License.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 */
package org.geotools.data.geoparquet;

import static java.lang.String.format;
import static java.util.Objects.requireNonNull;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.geotools.api.data.Transaction;
import org.geotools.api.feature.simple.SimpleFeatureType;
import org.geotools.api.feature.type.GeometryDescriptor;
import org.geotools.api.feature.type.Name;
import org.geotools.api.referencing.FactoryException;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;
import org.geotools.data.duckdb.DuckDBDialect;
import org.geotools.data.jdbc.FilterToSQL;
import org.geotools.data.store.ContentEntry;
import org.geotools.feature.AttributeTypeBuilder;
import org.geotools.feature.simple.SimpleFeatureTypeBuilder;
import org.geotools.geometry.jts.ReferencedEnvelope;
import org.geotools.jackson.datatype.geoparquet.GeoParquetMetadata;
import org.geotools.jdbc.AutoGeneratedPrimaryKeyColumn;
import org.geotools.jdbc.JDBCDataStore;
import org.geotools.jdbc.PrimaryKey;
import org.geotools.jdbc.PrimaryKeyColumn;
import org.geotools.jdbc.PrimaryKeyFinder;
import org.geotools.jdbc.SQLDialect;
import org.geotools.referencing.CRS;
import org.geotools.util.factory.Hints;
import org.geotools.util.logging.Logging;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryCollection;

/**
 * SQL Dialect for GeoParquet format.
 *
 * <p>This dialect extends the base DuckDB dialect with GeoParquet-specific functionality:
 *
 * <ul>
 *   <li>Parsing and utilizing GeoParquet metadata from the "geo" field
 *   <li>Setting up appropriate SQL views for GeoParquet files
 *   <li>Optimizing spatial operations and bounds computations
 *   <li>Handling both local and remote (HTTP, S3) GeoParquet data access
 * </ul>
 *
 * <p>The dialect extracts and uses the GeoParquet specification metadata to provide improved performance for operations
 * like bounds computation and feature access. It supports both standard GeoParquet format (1.1.0) and development
 * versions (1.2.0-dev).
 *
 * <p>The dialect uses several performance optimizations:
 *
 * <ul>
 *   <li>Extracting bounds from GeoParquet metadata rather than computing them
 *   <li>Creating SQL views for consistent access to partitioned datasets
 *   <li>Using DuckDB's spatial functions for efficient querying
 *   <li>Maintaining a cache of metadata to avoid repeated parsing
 * </ul>
 *
 * <p>It works in conjunction with {@link GeoParquetViewManager} to handle Hive-partitioned datasets, exposing each
 * partition as a separate feature type.
 */
public class GeoParquetDialect extends DuckDBDialect {

    private static final Logger LOGGER = Logging.getLogger(GeoParquetDialect.class);

    private final GeoParquetViewManager viewManager;

    /** Cached GeoParquet metadata extracted from the dataset */
    private Map<String, GeoparquetDatasetMetadata> geoparquetMetadata = new ConcurrentHashMap<>();

    /**
     * Thread-local storage for the current feature type name being processed.
     *
     * <p>This thread-local variable is used to pass context information between different parts of the code that need
     * to know which feature type is being accessed. It's particularly important for:
     *
     * <ol>
     *   <li>SQL generation during feature access, to apply the right geometry type handling rules
     *   <li>Ensuring proper multi-geometry encoding for collection geometries
     *   <li>Feature iteration in OverridingFeatureCollection
     *   <li>Bridging the information gap when JDBCDataStore calls methods like encodeGeometryColumn() that only have
     *       geometry column information without feature type context
     * </ol>
     *
     * <p>Since JDBCDataStore and JDBCFeatureSource are final classes that can't be easily extended, this thread-local
     * approach provides a way to pass essential type information to dialect methods that otherwise wouldn't have access
     * to the feature type context.
     *
     * <p>The variable is set in GeoparquetDataStore's getSchema and getFeatureSource methods and in
     * OverridingFeatureCollection's features method, and cleared after use with remove() to avoid thread-local memory
     * leaks.
     */
    static final ThreadLocal<String> CURRENT_TYPENAME = new ThreadLocal<>();

    /**
     * Creates a new GeoParquetDialect.
     *
     * @param dataStore The JDBC datastore this dialect will work with
     */
    public GeoParquetDialect(JDBCDataStore dataStore) {
        super(dataStore);
        this.viewManager = new GeoParquetViewManager(dataStore);
    }

    /**
     * Ensures that a database view exists for the specified feature type.
     *
     * <p>This method is called before any operations that require access to a feature type's schema or data,
     * implementing the lazy initialization pattern. If the view already exists, this method has no effect.
     *
     * @param viewName The name of the view/feature type to ensure exists
     * @throws IOException If there is an error creating the view
     */
    public void ensureViewExists(String viewName) throws IOException {
        viewManager.createViewIfNotExists(viewName);
    }

    /**
     * Returns a list of all available feature type names.
     *
     * <p>This method queries the view manager to get the names of all registered views, which correspond to available
     * feature types in the GeoParquet dataset.
     *
     * @return A list of feature type names
     * @throws IOException If there is an error retrieving the names
     */
    public List<String> getTypeNames() throws IOException {
        return viewManager.getViewNames();
    }

    /**
     * Gets a database connection from the connection pool.
     *
     * @return A JDBC connection
     * @throws IOException If there is an error getting a connection
     */
    Connection getConnection() throws IOException {
        return viewManager.getConnection();
    }

    /**
     * Creates a specialized filter-to-SQL converter for GeoParquet.
     *
     * @return A new GeoParquetFilterToSQL instance
     */
    @Override
    public FilterToSQL createFilterToSQL() {
        return new GeoParquetFilterToSQL();
    }

    /**
     * Provides SQL statements to initialize the DuckDB database for GeoParquet access.
     *
     * <p>This installs and loads required DuckDB extensions:
     *
     * <ul>
     *   <li>httpfs - For HTTP/S3 access to remote GeoParquet files
     *   <li>parquet - For reading Parquet file format
     * </ul>
     *
     * @return List of SQL statements to initialize the database
     */
    @Override
    public List<String> getDatabaseInitSql() {
        List<String> initScript = new ArrayList<>(super.getDatabaseInitSql());
        initScript.add("install httpfs");
        initScript.add("load httpfs");
        initScript.add("install parquet");
        initScript.add("load parquet");
        return initScript;
    }

    /**
     * Registers SQL views for GeoParquet data partitions.
     *
     * <p>This method is called by {@link GeoParquetDataStoreFactory#setupDataStore(JDBCDataStore, Map)} to initialize
     * the dialect with the provided configuration. It:
     *
     * <ol>
     *   <li>Clears any cached metadata
     *   <li>Initializes the view manager with the new configuration
     * </ol>
     *
     * @param config The GeoParquet configuration
     * @throws IOException If there's an error registering the views
     */
    public void initialize(GeoParquetConfig config) throws IOException {
        geoparquetMetadata.clear();
        viewManager.initialize(config);
    }

    /**
     * Gets the GeoParquet metadata for a feature type.
     *
     * <p>This is a convenience method that creates a connection and delegates to {@link #getGeoparquetMetadata(String,
     * Connection)} if the metadata for {@code typeName} is not already cached.
     *
     * @param typeName The feature type to get metadata for
     * @return The GeoParquet metadata for the feature type
     * @throws IOException If there is an error retrieving the metadata
     */
    public GeoparquetDatasetMetadata getGeoparquetMetadata(String typeName) throws IOException {
        GeoparquetDatasetMetadata md = geoparquetMetadata.get(typeName);
        if (md == null) {
            try (Connection c = viewManager.getConnection()) {
                md = getGeoparquetMetadata(typeName, c);
            } catch (SQLException e) {
                throw new IOException(e);
            }
        }
        return md;
    }

    /**
     * Gets the GeoParquet metadata for a feature type.
     *
     * <p>This method retrieves metadata from the cache if available, or loads it from the data source if not yet
     * cached.
     *
     * @param featureType The feature type to get metadata for
     * @param cx Database connection to use if metadata needs to be loaded
     * @return The GeoParquet metadata for the feature type
     */
    GeoparquetDatasetMetadata getGeoparquetMetadata(String featureType, Connection cx) {
        return geoparquetMetadata.computeIfAbsent(featureType, view -> loadGeoparquetMetadata(featureType, cx));
    }

    /**
     * Loads GeoParquet metadata for a specific view.
     *
     * <p>This method:
     *
     * <ol>
     *   <li>Retrieves the URI for the view
     *   <li>Queries the Parquet key-value metadata to extract the 'geo' field
     *   <li>Parses the metadata for each file in the dataset
     * </ol>
     *
     * @param viewName The name of the view to load metadata for
     * @param cx Database connection to use for querying
     * @return The combined dataset metadata
     */
    public GeoparquetDatasetMetadata loadGeoparquetMetadata(String viewName, Connection cx) {

        String lookUpUri = viewManager.getVieUri(viewName);
        String sql = format(
                "SELECT file_name, decode(value) AS value FROM parquet_kv_metadata('%s') WHERE key = 'geo'", lookUpUri);

        Map<String, GeoParquetMetadata> parquetMetadataByFileName = new HashMap<>();
        try (Statement st = cx.createStatement();
                ResultSet rs = st.executeQuery(sql)) {
            while (rs.next()) {
                String fileName = rs.getString("file_name");
                String geo = rs.getString("value");
                GeoParquetMetadata md = parseMetadata(fileName, geo);
                if (md != null) {
                    parquetMetadataByFileName.put(fileName, md);
                }
            }
        } catch (SQLException e) {
            throw new IllegalStateException(e);
        }

        return new GeoparquetDatasetMetadata(parquetMetadataByFileName);
    }

    /**
     * Parses the 'geo' metadata blob from a Parquet file into a structured object.
     *
     * <p>This method:
     *
     * <ol>
     *   <li>Extracts the binary data from the blob
     *   <li>Converts it to a UTF-8 string (the geo metadata is stored as JSON)
     *   <li>Parses the JSON into a GeoParquetMetadata object
     * </ol>
     *
     * @param fileName The name of the file this metadata belongs to
     * @param geoMetadata The 'geo' metadata as a JSON string
     * @return The parsed metadata, or null if parsing failed
     */
    private GeoParquetMetadata parseMetadata(String fileName, String geoMetadata) {
        if (geoMetadata != null) {
            try {
                return GeoParquetMetadata.readValue(geoMetadata);
            } catch (IOException e) {
                LOGGER.log(
                        Level.SEVERE,
                        e,
                        () -> format("Error parsing geoparquet metadata. File: %s, geo: %s", fileName, geoMetadata));
            }
        }
        return null;
    }

    /**
     * Provides a PrimaryKeyFinder that identifies the 'id' column as the primary key.
     *
     * <p>This is a helper for {@link GeoParquetDataStoreFactory} to establish the feature ID column in GeoParquet
     * datasets. It always identifies the 'id' column as a String primary key, which is the standard convention for
     * GeoParquet files.
     *
     * @return A PrimaryKeyFinder for GeoParquet datasets
     */
    public PrimaryKeyFinder getPrimaryKeyFinder() {
        return new PrimaryKeyFinder() {
            @Override
            public PrimaryKey getPrimaryKey(JDBCDataStore store, String schema, String table, Connection cx) {
                List<PrimaryKeyColumn> columns = List.of(new AutoGeneratedPrimaryKeyColumn("id", String.class));
                return new PrimaryKey(table, columns);
            }
        };
    }

    /**
     * Returns optimized bounds for a feature type by using GeoParquet metadata.
     *
     * <p>This method uses a multi-stage approach to efficiently determine dataset bounds:
     *
     * <ol>
     *   <li>First tries to extract bounds from the GeoParquet 'geo' metadata field
     *   <li>If 'geo' metadata is not available, checks for a 'bbox' column and uses aggregate functions on its
     *       components (common in datasets like OvertureMaps)
     *   <li>Finally falls back to the generic DuckDB bounds computation using spatial functions
     * </ol>
     *
     * <p>Each method is progressively more computationally expensive, so we try them in order of efficiency.
     *
     * @param schema The database schema (unused in GeoParquet)
     * @param featureType The feature type to get bounds for
     * @param cx Database connection to use for querying
     * @return A list containing a single ReferencedEnvelope representing the dataset bounds
     * @throws SQLException If there's an error executing SQL
     * @throws IOException If there's an error accessing the data
     */
    @Override
    public List<ReferencedEnvelope> getOptimizedBounds(String schema, SimpleFeatureType featureType, Connection cx)
            throws SQLException, IOException {

        if (null == featureType.getGeometryDescriptor()) {
            return List.of();
        }

        GeoparquetDatasetMetadata md = getGeoparquetMetadata(featureType.getTypeName(), cx);
        ReferencedEnvelope bounds;
        if (!md.isEmpty()) {
            bounds = md.getBounds();
        } else if (featureType.getDescriptor("bbox") != null) {
            // no geoparquet metadata ('geo' key present), see if there's a 'bbox' column
            // anyways
            bounds = computeBoundsFromBboxColumn(featureType, cx);
        } else {
            // fall back to ST_Extent_Agg(geometry)
            bounds = super.optimizedBounds(featureType, cx);
        }
        return List.of(requireNonNull(bounds));
    }

    /**
     * Computes bounds from a 'bbox' column in the dataset.
     *
     * <p>This method handles datasets that don't have GeoParquet 'geo' metadata but do have a 'bbox' column with xmin,
     * xmax, ymin, ymax components. It uses SQL aggregate functions to efficiently compute the overall bounds without
     * having to examine each geometry.
     *
     * @param featureType The feature type containing a bbox column
     * @param cx Database connection to use for querying
     * @return The computed bounds as a ReferencedEnvelope
     * @throws SQLException If there's an error executing SQL
     * @throws IOException If there's an error accessing the data
     */
    ReferencedEnvelope computeBoundsFromBboxColumn(SimpleFeatureType featureType, Connection cx)
            throws SQLException, IOException {

        String sql = format(
                "SELECT ST_AsWKB(ST_MakeEnvelope(MIN(bbox.xmin), MIN(bbox.ymin), MAX(bbox.xmax), MAX(bbox.ymax))::GEOMETRY)::BLOB FROM %s",
                escapeName(featureType.getTypeName()));

        try (PreparedStatement ps = cx.prepareStatement(sql);
                ResultSet rs = ps.executeQuery()) {
            if (!rs.next()) throw new RuntimeException("Could not compute bounds from bbox column, no result found");
            Geometry fullBounds = parseWKB(rs.getBlob(1));
            GeometryDescriptor geometryDescriptor = featureType.getGeometryDescriptor();
            CoordinateReferenceSystem crs = geometryDescriptor.getCoordinateReferenceSystem();
            return new ReferencedEnvelope(fullBounds.getEnvelopeInternal(), crs);
        }
    }

    /**
     * Gets the SRID (Spatial Reference ID) for a geometry column.
     *
     * <p>This method attempts to extract the SRID from the GeoParquet metadata's CRS information:
     *
     * <ol>
     *   <li>First tries to get the CRS from the GeoParquet metadata for the specified column
     *   <li>If available, extracts the SRID from the CRS definition using the PROJJSON representation
     *   <li>Falls back to trying the primary geometry column if the specific column CRS is not found
     *   <li>Falls back to EPSG:4326 (WGS84) if the CRS information is not available or doesn't contain SRID
     * </ol>
     *
     * <p>The CRS information is extracted from the GeoParquet 'geo' metadata field, which follows the PROJJSON v0.7
     * schema as defined by the OGC GeoParquet specification. This includes proper handling of CRS identifiers with
     * authority and code properties.
     *
     * <p>The implementation supports strongly-typed CRS objects, converting between the PROJJSON format used in
     * GeoParquet files and GeoTools CoordinateReferenceSystem objects.
     *
     * @param schemaName The database schema (unused in GeoParquet)
     * @param tableName The table/view name
     * @param columnName The geometry column name
     * @param cx Database connection
     * @return The SRID of the geometry column (from metadata or 4326 as default)
     */
    @Override
    public Integer getGeometrySRID(String schemaName, String tableName, String columnName, Connection cx) {
        requireNonNull(tableName);
        requireNonNull(columnName);
        requireNonNull(cx);

        Integer srid = null;
        try {
            GeoparquetDatasetMetadata metadata = getGeoparquetMetadata(tableName, cx);
            srid = getGeometrySRIDInternal(metadata, columnName);
            if (srid == null) {
                // If specific column CRS not found, try the primary column
                CoordinateReferenceSystem crs = metadata.getCrs();
                srid = lookupEpsgCode(crs);
            }
        } catch (RuntimeException e) {
            LOGGER.log(Level.WARNING, "Error getting CRS from GeoParquet metadata", e);
        }

        if (srid == null) {
            srid = 4326; // Default if no metadata available
        }
        return srid;
    }

    /**
     * Override to use the {@link GeoParquetMetadata} provided axis order on a per-FeatureType basis.
     * {@link SQLDialect#createCRS} uses the {@link #forceLongitudeFirst} flag as a constant.
     */
    @Override
    public CoordinateReferenceSystem createCRS(int srid, Connection cx) throws SQLException {
        String typeName = CURRENT_TYPENAME.get();
        if (typeName != null) {
            // note we're abusing the fact that this method is only ever called right after getGeometrySRID(), so we do
            // know we're being asked for the declared CRS and not any random one
            GeoparquetDatasetMetadata md = getGeoparquetMetadata(typeName, cx);
            CoordinateReferenceSystem crs = md.getCrs();
            // but check it anyway
            try {
                Integer id = CRS.lookupEpsgCode(crs, false);
                if (id != null && id.intValue() == srid) {
                    return crs;
                }
            } catch (FactoryException e) {
                LOGGER.log(
                        Level.FINE, "Could not figure out CRS id, proceeding with regular parsing of the provided id");
            }
        }

        boolean longitudeFirst = true; // default to x/y
        try {
            return CRS.decode("EPSG:" + srid, longitudeFirst);
        } catch (Exception e) {
            LOGGER.log(Level.FINE, "Could not decode EPSG:" + srid + " using the EPSG plugins.");
            return null;
        }
    }

    /**
     * Encodes a geometry column for a SQL query with awareness of geometry types.
     *
     * <p>This overridden method enhances the base DuckDB dialect implementation by checking if multi-geometry encoding
     * should be enforced for the current feature type. It uses the CURRENT_TYPENAME thread-local variable to determine
     * the appropriate behavior based on the GeoParquet metadata.
     *
     * <p>For example, if the geometry column is a MultiPolygon according to the GeoParquet metadata, this method will
     * add ST_Multi() to the SQL encoding to ensure proper handling of collection geometries. This is crucial because
     * the JDBCDataStore calls this method without providing full feature type context.
     *
     * @param gatt The geometry descriptor to encode
     * @param prefix Column prefix to use
     * @param srid The spatial reference ID
     * @param hints Rendering hints that may affect encoding
     * @param sql The SQL buffer to append to
     */
    @Override
    public void encodeGeometryColumn(GeometryDescriptor gatt, String prefix, int srid, Hints hints, StringBuffer sql) {
        boolean forceMulti = shouldEnforceMuliForCurrentTypeName(gatt.getLocalName());
        encodeGeometryColumnInternal(gatt, prefix, hints, forceMulti, sql);
    }

    /**
     * Creates a new feature type with more specific geometry types based on GeoParquet metadata, with results cached
     * for performance.
     *
     * <p>This method processes a feature type to enhance its geometry descriptors with more specific geometry types
     * derived from the GeoParquet metadata. It:
     *
     * <ol>
     *   <li>Ensures the database view exists for the feature type
     *   <li>Delegates to the GeoParquetViewManager to check for a cached version of the enhanced schema
     *   <li>If no cached version exists, creates a new schema with correct geometry type bindings
     *   <li>Caches the result for future use
     * </ol>
     *
     * <p>This is essential because DuckDB only reports a generic GEOMETRY type, while the GeoParquet metadata contains
     * information about the actual geometry types (Point, LineString, etc.).
     *
     * <p>The caching mechanism improves performance by avoiding repeated metadata lookups and feature type construction
     * while maintaining thread safety through the GeoParquetViewManager.
     *
     * @param schema The original feature type with generic geometry types
     * @return A new feature type with more specific geometry types, either freshly built or from cache
     * @throws IOException If there is an error accessing the GeoParquet metadata
     */
    public SimpleFeatureType fixGeometryTypes(SimpleFeatureType schema) throws IOException {
        ensureViewExists(schema.getTypeName());

        SimpleFeatureType featureType;
        try {
            featureType = viewManager.getViewFeatureType(schema, this::buildTypeNarrowedGeometryFeatureType);
        } catch (UncheckedIOException e) {
            throw e.getCause();
        }
        Name name = featureType.getName();
        ContentEntry entry = dataStore.getEntry(name);
        entry.getState(Transaction.AUTO_COMMIT).setFeatureType(featureType);
        return featureType;
    }

    /**
     * Creates a feature type with narrowed geometry types based on GeoParquet metadata.
     *
     * <p>This method is called by fixGeometryTypes via the GeoParquetViewManager's caching mechanism. It performs the
     * actual work of building an enhanced feature type by:
     *
     * <ol>
     *   <li>Creating a new feature type builder initialized with the original schema
     *   <li>Processing all geometry descriptors to determine their specific types
     *   <li>Replacing each geometry descriptor with one that has the correct binding
     *   <li>Building the new feature type with the enhanced descriptors
     *   <li>Updating the content entry's state for consistency
     * </ol>
     *
     * <p>This is the core functionality that translates GeoParquet metadata into proper GeoTools feature type
     * definitions with correct geometry types.
     *
     * @param schema The original feature type with generic geometry types
     * @return A new feature type with more specific geometry types
     */
    private SimpleFeatureType buildTypeNarrowedGeometryFeatureType(SimpleFeatureType schema) {
        SimpleFeatureTypeBuilder builder = new SimpleFeatureTypeBuilder();
        builder.init(schema);

        schema.getAttributeDescriptors().stream()
                .filter(GeometryDescriptor.class::isInstance)
                .map(GeometryDescriptor.class::cast)
                .map(d -> buildGeometryDescriptorOverride(schema.getTypeName(), d))
                .forEach(overriding -> builder.set(overriding.getLocalName(), overriding));
        return builder.buildFeatureType();
    }

    /**
     * Creates a geometry descriptor with the correct geometry type for a column.
     *
     * <p>This method creates a new geometry descriptor with a more specific geometry type binding based on the
     * GeoParquet metadata. It:
     *
     * <ol>
     *   <li>Fetches the GeoParquet metadata for the feature type
     *   <li>Determines the most specific geometry type for the column using getNarrowedGeometryType
     *   <li>If the type is different from the original descriptor's binding, creates a new descriptor
     *   <li>Otherwise, returns the original descriptor unchanged
     * </ol>
     *
     * <p>This allows for proper handling of specific geometry types (Point, LineString, etc.) even though DuckDB only
     * reports a generic GEOMETRY type.
     *
     * @param typeName The name of the feature type containing the geometry column
     * @param orig The original geometry descriptor with generic type
     * @return A new geometry descriptor with the specific geometry type, or the original if unchanged
     */
    private GeometryDescriptor buildGeometryDescriptorOverride(String typeName, GeometryDescriptor orig) {
        String geometryAttirbute = orig.getLocalName();
        Class<? extends Geometry> narrowedGeomType = getNarrowedGeometryType(typeName, geometryAttirbute);
        if (narrowedGeomType.equals(orig.getType().getBinding())) {
            return orig;
        }

        AttributeTypeBuilder builder = new AttributeTypeBuilder();
        builder.init(orig);
        builder.setBinding(narrowedGeomType);
        return (GeometryDescriptor) builder.buildDescriptor(geometryAttirbute);
    }

    /**
     * Gets the narrowed geometry type for a specific column in a feature type.
     *
     * <p>This method determines the most specific geometry type for a column by:
     *
     * <ol>
     *   <li>Fetching the GeoParquet metadata for the feature type
     *   <li>Delegating to GeoparquetDatasetMetadata.getNarrowedGeometryType() to analyze the metadata and determine the
     *       appropriate JTS geometry class
     * </ol>
     *
     * <p>This is a helper method used by other methods that need to work with specific geometry types, such as
     * buildGeometryDescriptorOverride and enforceMuliForCurrentTypeName.
     *
     * @param typeName The name of the feature type
     * @param geometryAttirbute The name of the geometry column
     * @return The most specific JTS geometry class for the column
     */
    private Class<? extends Geometry> getNarrowedGeometryType(String typeName, String geometryAttirbute) {
        GeoparquetDatasetMetadata md;
        try {
            md = getGeoparquetMetadata(typeName);
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        return md.getNarrowedGeometryType(geometryAttirbute);
    }

    /**
     * Determines if multi-geometry encoding should be enforced for the current feature type.
     *
     * <p>This method checks if the geometry type for the current feature type (available via the CURRENT_TYPENAME
     * thread-local variable) is a GeometryCollection or subclass. If so, it indicates that SQL generation should force
     * the use of ST_Multi to ensure proper handling of collection geometries.
     *
     * <p>This is important because:
     *
     * <ol>
     *   <li>JDBCDataStore calls encodeGeometryColumn() without providing full feature type context
     *   <li>DuckDB's spatial functions don't automatically preserve collection types
     * </ol>
     *
     * <p>The thread-local CURRENT_TYPENAME provides the necessary context to determine whether a geometry column needs
     * multi-geometry treatment in the generated SQL.
     *
     * @param geometryAttribute The name of the geometry attribute to check
     * @return true if multi-geometry encoding should be enforced, false otherwise
     */
    private boolean shouldEnforceMuliForCurrentTypeName(String geometryAttribute) {
        String typeName = CURRENT_TYPENAME.get();
        if (typeName == null) {
            return false;
        }
        Class<? extends Geometry> type = getNarrowedGeometryType(typeName, geometryAttribute);
        return GeometryCollection.class.isAssignableFrom(type);
    }

    /**
     * Internal helper method to get the SRID from metadata for a specific column.
     *
     * @param metadata The GeoParquet dataset metadata
     * @return The SRID if found, or null if not available
     */
    Integer getGeometrySRIDInternal(GeoparquetDatasetMetadata metadata, String columnName) {
        Integer srid = null;

        // Try to get the specific column's CRS if columnName is provided
        // Check if this is the geometry column from the metadata
        if (metadata.getColumn(columnName).isPresent()) {
            CoordinateReferenceSystem crs = metadata.getCrs(columnName);
            srid = lookupEpsgCode(crs);
        }

        return srid;
    }

    /**
     * Attempts to find an EPSG code for the given coordinate reference system.
     *
     * @param crs The coordinate reference system
     * @return The EPSG code if found, or null if not found or an error occurred
     */
    private Integer lookupEpsgCode(CoordinateReferenceSystem crs) {
        Integer srid = null;
        if (crs != null) {
            try {
                srid = CRS.lookupEpsgCode(crs, true);
            } catch (FactoryException e) {
                LOGGER.log(Level.FINE, "Could not determine EPSG code from CRS", e);
            }
        }
        return srid;
    }
}
