/*
 *    GeoTools - The Open Source Java GIS Toolkit
 *    http://geotools.org
 *
 *    (C) 2001-2008, Open Source Geospatial Foundation (OSGeo)
 *    (C) 2001-2006  Vivid Solutions
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation;
 *    version 2.1 of the License.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 */
package org.geotools.geometry.iso.topograph2D.index;

/**
 * MonotoneChains are a way of partitioning the segments of an edge to allow for fast searching of
 * intersections. They have the following properties:
 *
 * <ol>
 *   <li>the segments within a monotone chain will never intersect each other
 *   <li>the envelope of any contiguous subset of the segments in a monotone chain is simply the
 *       envelope of the endpoints of the subset.
 * </ol>
 *
 * Property 1 means that there is no need to test pairs of segments from within the same monotone
 * chain for intersection. Property 2 allows binary search to be used to find the intersection
 * points of two monotone chains. For many types of real-world data, these properties eliminate a
 * large number of segment comparisons, producing substantial speed gains.
 */
public class MonotoneChain {

    MonotoneChainEdge mce;

    int chainIndex;

    public MonotoneChain(MonotoneChainEdge mce, int chainIndex) {
        this.mce = mce;
        this.chainIndex = chainIndex;
    }

    public void computeIntersections(MonotoneChain mc, SegmentIntersector si) {
        this.mce.computeIntersectsForChain(chainIndex, mc.mce, mc.chainIndex, si);
    }
}
