/*
 * Decompiled with CFR 0.152.
 */
package org.geotools.data.elasticsearch;

import com.github.davidmoten.geo.GeoHash;
import com.github.davidmoten.geo.LatLong;
import java.awt.geom.Point2D;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.geotools.data.elasticsearch.FilterToElasticHelper;
import org.geotools.referencing.GeodeticCalculator;
import org.geotools.referencing.datum.DefaultEllipsoid;
import org.geotools.util.logging.Logging;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Envelope;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.LinearRing;
import org.locationtech.jts.geom.Polygon;
import org.locationtech.jts.io.ParseException;
import org.locationtech.jts.io.WKTReader;
import org.opengis.referencing.datum.Ellipsoid;

class ElasticParserUtil {
    private static final Logger LOGGER = Logging.getLogger(ElasticParserUtil.class);
    private static final Pattern GEO_POINT_PATTERN = Pattern.compile("\\s*([-+]?\\d*\\.?\\d*)[^-+\\d.]+([-+]?\\d*\\.?\\d*)\\s*");
    private static final Pattern GEO_HASH_PATTERN = Pattern.compile("[0123456789bcdefghjkmnpqrstuvwxyz]+");
    private static final Pattern ELASTIC_DISTANCE_PATTERN = Pattern.compile("([0-9]+(\\.[0-9]+)?)([a-zA-Z]*)");
    private static final double CIRCLE_INTERPOLATION_INTERVAL = 500.0;
    private static final int MAX_CIRCLE_POINTS = 500;
    private static final int MIN_CIRCLE_POINTS = 40;
    private static final double MIN_CIRCLE_RADIUS_M = 0.001;
    private final GeodeticCalculator geodeticCalculator;
    private static final Pattern WKT_PATTERN = Pattern.compile("POINT.*|LINESTRING.*|POLYGON.*|MULTIPOINT.*|MULTILINESTRING.*|MULTIPOLYGON.*|GEOMETRYCOLLECTION.*");
    private final GeometryFactory geometryFactory = new GeometryFactory();
    private final WKTReader wktReader;

    public ElasticParserUtil() {
        this.geodeticCalculator = new GeodeticCalculator((Ellipsoid)DefaultEllipsoid.WGS84);
        this.wktReader = new WKTReader();
    }

    public Geometry createGeometry(Object obj) {
        Geometry geometry;
        if (obj instanceof String) {
            Matcher listMatcher = GEO_POINT_PATTERN.matcher((String)obj);
            if (listMatcher.matches()) {
                double y = Double.parseDouble(listMatcher.group(1));
                double x = Double.parseDouble(listMatcher.group(2));
                geometry = this.geometryFactory.createPoint(new Coordinate(x, y));
            } else if (GEO_HASH_PATTERN.matcher((String)obj).matches()) {
                LatLong latLon = GeoHash.decodeHash((String)((String)obj));
                Coordinate geoPoint = new Coordinate(latLon.getLon(), latLon.getLat());
                double lat = geoPoint.y;
                double lon = geoPoint.x;
                geometry = this.geometryFactory.createPoint(new Coordinate(lon, lat));
            } else if (WKT_PATTERN.matcher((String)obj).matches()) {
                Geometry geom;
                try {
                    geom = this.wktReader.read((String)obj);
                }
                catch (ParseException e) {
                    geom = null;
                }
                geometry = geom;
            } else {
                geometry = null;
            }
        } else if (obj instanceof List && ((List)obj).size() == 2) {
            List values = (List)obj;
            if (Number.class.isAssignableFrom(values.get(0).getClass())) {
                double x = ((Number)values.get(0)).doubleValue();
                double y = ((Number)values.get(1)).doubleValue();
                geometry = this.geometryFactory.createPoint(new Coordinate(x, y));
            } else if (values.get(0) instanceof String) {
                double x = Double.parseDouble((String)values.get(0));
                double y = Double.parseDouble((String)values.get(1));
                geometry = this.geometryFactory.createPoint(new Coordinate(x, y));
            } else {
                geometry = null;
            }
        } else {
            geometry = obj instanceof Map ? this.createGeometry((Map)obj) : null;
        }
        return geometry;
    }

    public Geometry createGeometry(Map<String, Object> properties) {
        Polygon geometry;
        switch (String.valueOf(properties.get("type")).toUpperCase()) {
            case "POINT": {
                List posList = (List)properties.get("coordinates");
                Coordinate coordinate = this.createCoordinate(posList);
                geometry = this.geometryFactory.createPoint(coordinate);
                break;
            }
            case "LINESTRING": {
                List posList = (List)properties.get("coordinates");
                Coordinate[] coordinates = this.createCoordinates(posList);
                geometry = this.geometryFactory.createLineString(coordinates);
                break;
            }
            case "POLYGON": {
                List posList = (List)properties.get("coordinates");
                geometry = this.createPolygon(posList);
                break;
            }
            case "MULTIPOINT": {
                List posList = (List)properties.get("coordinates");
                Coordinate[] coordinates = this.createCoordinates(posList);
                geometry = this.geometryFactory.createMultiPointFromCoords(coordinates);
                break;
            }
            case "MULTILINESTRING": {
                List posList = (List)properties.get("coordinates");
                LineString[] lineStrings = new LineString[posList.size()];
                for (int i = 0; i < posList.size(); ++i) {
                    Coordinate[] coordinates = this.createCoordinates((List)posList.get(i));
                    lineStrings[i] = this.geometryFactory.createLineString(coordinates);
                }
                geometry = this.geometryFactory.createMultiLineString(lineStrings);
                break;
            }
            case "MULTIPOLYGON": {
                List posList = (List)properties.get("coordinates");
                Polygon[] polygons = new Polygon[posList.size()];
                for (int i = 0; i < posList.size(); ++i) {
                    polygons[i] = this.createPolygon((List)posList.get(i));
                }
                geometry = this.geometryFactory.createMultiPolygon(polygons);
                break;
            }
            case "GEOMETRYCOLLECTION": {
                List list = (List)properties.get("geometries");
                Geometry[] geometries = new Geometry[list.size()];
                for (int i = 0; i < geometries.length; ++i) {
                    geometries[i] = this.createGeometry((Map)list.get(i));
                }
                geometry = this.geometryFactory.createGeometryCollection(geometries);
                break;
            }
            case "ENVELOPE": {
                List posList = (List)properties.get("coordinates");
                Coordinate[] coords = this.createCoordinates(posList);
                Envelope envelope = new Envelope(coords[0], coords[1]);
                geometry = this.geometryFactory.toGeometry(envelope);
                break;
            }
            case "CIRCLE": {
                List posList = (List)properties.get("coordinates");
                String radius = (String)properties.get("radius");
                Coordinate coordinate = this.createCoordinate(posList);
                geometry = this.createCircle(coordinate, radius);
                break;
            }
            default: {
                Object latObj = properties.get("lat");
                Object lonObj = properties.get("lon");
                if (latObj != null && lonObj != null) {
                    Double lat = latObj instanceof Number ? Double.valueOf(((Number)latObj).doubleValue()) : (latObj instanceof String ? Double.valueOf(Double.parseDouble((String)latObj)) : null);
                    Double lon = lonObj instanceof Number ? Double.valueOf(((Number)lonObj).doubleValue()) : (lonObj instanceof String ? Double.valueOf(Double.parseDouble((String)lonObj)) : null);
                    if (lat != null && lon != null) {
                        geometry = this.geometryFactory.createPoint(new Coordinate(lon.doubleValue(), lat.doubleValue()));
                        break;
                    }
                    geometry = null;
                    break;
                }
                geometry = null;
            }
        }
        return geometry;
    }

    private Polygon createPolygon(List<List<List<Object>>> posList) {
        Coordinate[] shellCoordinates = this.createCoordinates(posList.get(0));
        LinearRing shell = this.geometryFactory.createLinearRing(shellCoordinates);
        LinearRing[] holes = new LinearRing[posList.size() - 1];
        for (int i = 1; i < posList.size(); ++i) {
            Coordinate[] coordinates = this.createCoordinates(posList.get(i));
            holes[i - 1] = this.geometryFactory.createLinearRing(coordinates);
        }
        return this.geometryFactory.createPolygon(shell, holes);
    }

    private Coordinate[] createCoordinates(List<List<Object>> posList) {
        Coordinate[] coordinates = new Coordinate[posList.size()];
        for (int i = 0; i < posList.size(); ++i) {
            coordinates[i] = this.createCoordinate(posList.get(i));
        }
        return coordinates;
    }

    private Coordinate createCoordinate(List<Object> posList) {
        double y;
        double x;
        if (posList == null) {
            return null;
        }
        if (Number.class.isAssignableFrom(posList.get(0).getClass())) {
            x = ((Number)posList.get(0)).doubleValue();
            y = ((Number)posList.get(1)).doubleValue();
        } else {
            x = Double.parseDouble(posList.get(0).toString());
            y = Double.parseDouble(posList.get(1).toString());
        }
        return new Coordinate(x, y);
    }

    public List<Object> readField(Map<String, Object> source, String name) {
        List<String> keys = Arrays.asList(name.split("\\."));
        ArrayList<Object> values = new ArrayList<Object>();
        if (!keys.isEmpty()) {
            Object entry = source.get(keys.get(0));
            if (entry == null) {
                this.readField(source.get(name), new ArrayList<String>(), values);
            } else {
                this.readField(entry, keys.subList(1, keys.size()), values);
            }
        }
        ArrayList<Object> result = !values.isEmpty() ? values : null;
        return result;
    }

    private void readField(Object entry, List<String> keys, List<Object> values) {
        if (entry != null && List.class.isAssignableFrom(entry.getClass())) {
            for (Object object : (List)entry) {
                this.readField(object, keys, values);
            }
        } else if (entry != null && !keys.isEmpty() && Map.class.isAssignableFrom(entry.getClass())) {
            Object nextEntry = ((Map)entry).get(keys.get(0));
            List<String> newKeys = keys.subList(1, keys.size());
            this.readField(nextEntry, newKeys, values);
        } else if (entry != null) {
            values.add(entry);
        }
    }

    public static boolean isGeoPointFeature(Map<String, Object> map) {
        boolean result = false;
        if (map.size() == 2 && map.containsKey("coordinates")) {
            try {
                result = "geo_point".equals(((Map)map.get("coordinates")).get("type"));
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
        return result;
    }

    public static String urlDecode(String value) {
        try {
            value = URLDecoder.decode(value, StandardCharsets.UTF_8.toString());
        }
        catch (UnsupportedEncodingException e) {
            LOGGER.warning("Unable to encode value(s): " + String.valueOf(e));
        }
        return value;
    }

    private Geometry createCircle(Coordinate centreCoord, String radius) {
        double radM;
        if (centreCoord == null) {
            return null;
        }
        try {
            radM = ElasticParserUtil.convertToMeters(radius);
        }
        catch (Exception e) {
            return null;
        }
        if (radM < 0.001) {
            return null;
        }
        double circumferance = radM * 2.0 * Math.PI;
        int numPoints = (int)(circumferance / 500.0);
        numPoints = Math.max(40, numPoints);
        numPoints = Math.min(500, numPoints);
        double angularIncrement = 360.0 / (double)numPoints;
        this.geodeticCalculator.setStartingGeographicPoint(centreCoord.x, centreCoord.y);
        Coordinate[] linearRingCoords = new Coordinate[numPoints + 1];
        double angle = 0.0;
        for (int i = 0; i < numPoints; ++i) {
            this.geodeticCalculator.setDirection(angle, radM);
            Point2D point2D = this.geodeticCalculator.getDestinationGeographicPoint();
            linearRingCoords[i] = new Coordinate(point2D.getX(), point2D.getY());
            angle += angularIncrement;
        }
        linearRingCoords[numPoints] = linearRingCoords[0];
        LinearRing linearRing = this.geometryFactory.createLinearRing(linearRingCoords);
        return this.geometryFactory.createPolygon(linearRing);
    }

    static double convertToMeters(String distanceWithUnit) throws IllegalArgumentException {
        if (distanceWithUnit == null || distanceWithUnit.isEmpty()) {
            throw new IllegalArgumentException("Null of zero length distance string argument");
        }
        Matcher matcher = ELASTIC_DISTANCE_PATTERN.matcher(distanceWithUnit);
        if (matcher.matches()) {
            double distance = Double.parseDouble(matcher.group(1));
            String unit = matcher.group(3);
            Double conversion = FilterToElasticHelper.UNITS_MAP.get(unit);
            if (conversion == null) {
                if (unit != null && !unit.isEmpty()) {
                    throw new IllegalArgumentException("Illegal unit: " + unit);
                }
                conversion = 1.0;
            }
            return distance * conversion;
        }
        throw new IllegalArgumentException("Distance string argument has incorrect format");
    }
}

