/*
 *    GeoTools - The Open Source Java GIS Toolkit
 *    http://geotools.org
 *
 *    (C) 2009-2011, Open Source Geospatial Foundation (OSGeo)
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation;
 *    version 2.1 of the License.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 */
package org.geotools.data.complex;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.IOException;
import java.io.Serializable;
import java.net.URL;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;
import org.geotools.api.data.DataAccess;
import org.geotools.api.data.DataAccessFinder;
import org.geotools.api.data.DataSourceException;
import org.geotools.api.data.FeatureSource;
import org.geotools.api.feature.Feature;
import org.geotools.api.feature.type.FeatureType;
import org.geotools.api.feature.type.Name;
import org.geotools.data.complex.config.AppSchemaDataAccessConfigurator;
import org.geotools.data.complex.config.AppSchemaDataAccessDTO;
import org.geotools.data.complex.config.AttributeMapping;
import org.geotools.data.complex.config.NonFeatureTypeProxy;
import org.geotools.data.complex.config.SourceDataStore;
import org.geotools.data.complex.config.TypeMapping;
import org.geotools.data.complex.feature.type.Types;
import org.geotools.test.AppSchemaTestSupport;
import org.geotools.util.URLs;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * This tests AppSchemaDataAccessRegistry class. When an appschema data access is created, it would be registered in the
 * registry. Once it's in the registry, its feature type mapping and feature source (simple or mapped) would be
 * accessible globally.
 *
 * @author Rini Angreani (CSIRO Earth Science and Resource Engineering)
 */
public class AppSchemaDataAccessRegistryTest extends AppSchemaTestSupport {

    public static final Logger LOGGER =
            org.geotools.util.logging.Logging.getLogger(AppSchemaDataAccessRegistryTest.class);

    private static final String GSMLNS = "urn:cgi:xmlns:CGI:GeoSciML:2.0";

    private static final Name MAPPED_FEATURE = Types.typeName(GSMLNS, "MappedFeature");

    private static final Name GEOLOGIC_UNIT = Types.typeName("myGeologicUnit");

    private static final Name COMPOSITION_PART = Types.typeName(GSMLNS, "CompositionPart");

    private static final Name CGI_TERM_VALUE = Types.typeName(GSMLNS, "CGI_TermValue");

    private static final Name CONTROLLED_CONCEPT = Types.typeName(GSMLNS, "ControlledConcept");

    private static final String schemaBase = "/test-data/";

    /** Geological unit data access */
    private static AppSchemaDataAccess guDataAccess;

    /** Compositional part data access */
    private static AppSchemaDataAccess cpDataAccess;

    /** Mapped feature data access */
    private static AppSchemaDataAccess mfDataAccess;

    /** CGI Term Value data access */
    private static AppSchemaDataAccess cgiDataAccess;

    /** Controlled Concept data access */
    private static AppSchemaDataAccess ccDataAccess;

    /** App-schema FeatureTypeMapping extract from a mapping file. This one has a mappingName to identify itself. */
    private static TypeMapping dtoMappingName;

    /**
     * App-schema FeatureTypeMapping extract from a mapping file.This one has no mappingName, therefore targetElement is
     * used to identify itself.
     */
    private static TypeMapping dtoNoMappingName;

    /** App-schema mapping file extract. */
    private static AppSchemaDataAccessDTO config;

    /** Test registering and unregistering all data accesses works. */
    @Test
    public void testRegisterAndUnregisterDataAccess() throws Exception {
        loadDataAccesses();
        /** Check that data access are registered */
        this.checkRegisteredDataAccess(mfDataAccess, MAPPED_FEATURE, false);
        this.checkRegisteredDataAccess(guDataAccess, GEOLOGIC_UNIT, false);
        this.checkRegisteredDataAccess(cpDataAccess, COMPOSITION_PART, true);
        this.checkRegisteredDataAccess(cgiDataAccess, CGI_TERM_VALUE, true);
        this.checkRegisteredDataAccess(ccDataAccess, CONTROLLED_CONCEPT, true);

        /** Now unregister, and see if they're successful */
        unregister(mfDataAccess, MAPPED_FEATURE);
        unregister(guDataAccess, GEOLOGIC_UNIT);
        unregister(cpDataAccess, COMPOSITION_PART);
        unregister(cgiDataAccess, CGI_TERM_VALUE);
        unregister(ccDataAccess, CONTROLLED_CONCEPT);
    }

    /**
     * Test that asking for a nonexistent type causes an excception to be thrown with the correct number of type names
     * in the detail message.
     */
    @Test
    public void testThrowDataSourceException() throws Exception {
        Name typeName = Types.typeName(GSMLNS, "DoesNotExist");
        boolean handledException = false;
        try {
            AppSchemaDataAccessRegistry.getMappingByElement(typeName);
        } catch (DataSourceException e) {
            LOGGER.info(e.toString());
            handledException = true;
            assertTrue(e.getMessage().startsWith("Feature type " + typeName + " not found"));
        }
        assertTrue("Expected a DataSourceException to have been thrown and handled", handledException);
    }

    /** Load all data accesses */
    public static void loadDataAccesses() throws Exception {
        /** Load Mapped Feature data access */
        Map<String, Serializable> dsParams = new HashMap<>();
        URL url = AppSchemaDataAccessRegistryTest.class.getResource(schemaBase + "MappedFeaturePropertyfile.xml");
        assertNotNull(url);
        dsParams.put("dbtype", "app-schema");
        dsParams.put("url", url.toExternalForm());
        mfDataAccess = (AppSchemaDataAccess) DataAccessFinder.getDataStore(dsParams);
        assertNotNull(mfDataAccess);

        /** Load Geological Unit data access */
        url = AppSchemaDataAccessRegistryTest.class.getResource(schemaBase + "GeologicUnit.xml");
        assertNotNull(url);
        dsParams.put("url", url.toExternalForm());
        guDataAccess = (AppSchemaDataAccess) DataAccessFinder.getDataStore(dsParams);
        assertNotNull(guDataAccess);

        /** Find Compositional Part data access */
        cpDataAccess = (AppSchemaDataAccess) DataAccessRegistry.getDataAccess(COMPOSITION_PART);
        assertNotNull(cpDataAccess);

        /** Find CGI Term Value data access */
        cgiDataAccess = (AppSchemaDataAccess) DataAccessRegistry.getDataAccess(CGI_TERM_VALUE);
        assertNotNull(cgiDataAccess);

        /** Find ControlledConcept data access */
        ccDataAccess = (AppSchemaDataAccess) DataAccessRegistry.getDataAccess(CONTROLLED_CONCEPT);
        assertNotNull(ccDataAccess);
    }

    /** Create mock app-schema data access config. */
    @BeforeClass
    public static void oneTimeSetUp() {
        /** Create mock AppSchemaDataAccessDto to test mappingName */
        final String TARGET_ELEMENT_NAME = "gsml:MappedFeature";
        final String MAPPING_NAME = "MAPPING_NAME_ONE";
        final String SOURCE_ID = "MappedFeature";
        final String MAPPING_FILE = "MappedFeaturePropertyfile";

        Map<String, Serializable> dsParams = new HashMap<>();
        URL url = AppSchemaDataAccessRegistryTest.class.getResource(schemaBase);
        assertNotNull(url);
        final SourceDataStore ds = new SourceDataStore();
        ds.setId(SOURCE_ID);
        dsParams.put("directory", URLs.urlToFile(url).getPath());
        ds.setParams(dsParams);
        config = new AppSchemaDataAccessDTO();
        config.setSourceDataStores(List.of(ds));
        config.setBaseSchemasUrl(url.toExternalForm());
        config.setNamespaces(Map.of("gsml", GSMLNS));
        config.setTargetSchemasUris(List.of("http://www.geosciml.org/geosciml/2.0/xsd/geosciml.xsd"));
        config.setCatalog("mappedPolygons.oasis.xml");

        /** Create mock TypeMapping objects to be set inside config in the test cases */
        dtoMappingName = new TypeMapping();
        dtoMappingName.setMappingName(MAPPING_NAME);
        dtoMappingName.setSourceDataStore(SOURCE_ID);
        dtoMappingName.setSourceTypeName(MAPPING_FILE);
        dtoMappingName.setTargetElementName(TARGET_ELEMENT_NAME);

        dtoNoMappingName = new TypeMapping();
        dtoNoMappingName.setSourceDataStore(SOURCE_ID);
        dtoNoMappingName.setSourceTypeName(MAPPING_FILE);
        dtoNoMappingName.setTargetElementName(TARGET_ELEMENT_NAME);
    }

    /**
     * Tests that registry works.
     *
     * @param dataAccess The app schema data access to check
     * @param typeName Feature type
     * @param isNonFeature true if the type is non feature
     */
    private void checkRegisteredDataAccess(AppSchemaDataAccess dataAccess, Name typeName, boolean isNonFeature)
            throws IOException {
        FeatureTypeMapping mapping = AppSchemaDataAccessRegistry.getMappingByName(typeName);
        assertNotNull(mapping);
        // compare with the supplied data access
        assertEquals(dataAccess.getMappingByName(typeName), mapping);
        if (isNonFeature) {
            assertTrue(mapping.getTargetFeature().getType() instanceof NonFeatureTypeProxy);
        }

        // should return a simple feature source
        FeatureSource source =
                AppSchemaDataAccessRegistry.getMappingByName(typeName).getSource();
        assertNotNull(source);
        assertEquals(mapping.getSource(), source);

        // should return a mapping feature source
        FeatureSource<FeatureType, Feature> mappedSource = DataAccessRegistry.getFeatureSource(typeName);
        assertNotNull(mappedSource);
        // compare with the supplied data access
        assertEquals(mappedSource.getDataStore(), dataAccess);
    }

    /**
     * Tests that unregistering data access works
     *
     * @param dataAccess The data access
     * @param typeName The feature type name
     */
    private void unregister(DataAccess dataAccess, Name typeName) throws IOException {
        dataAccess.dispose();
        boolean notFound = false;
        try {
            AppSchemaDataAccessRegistry.getMappingByElement(typeName);
        } catch (DataSourceException e) {
            notFound = true;
            assertTrue(e.getMessage().startsWith("Feature type " + typeName + " not found"));
        }
        if (!notFound) {
            fail("Expecting DataSourceException but didn't occur. Deregistering data access fails.");
        }
        notFound = false;
        try {
            AppSchemaDataAccessRegistry.getSimpleFeatureSource(typeName);
        } catch (DataSourceException e) {
            notFound = true;
            assertTrue(e.getMessage().startsWith("Feature type " + typeName + " not found"));
        }
        if (!notFound) {
            fail("Expecting DataSourceException but didn't occur. Deregistering data access fails.");
        }
    }

    /** Fail scenarios for breaking uniqueness of FeatureTypeMapping key (mappingName or targetElement). */
    @Test
    public void testDuplicateKey() throws IOException {
        boolean threwException = false;
        /** Test duplicate mappingName */
        Set<TypeMapping> mappings = new HashSet<>();
        TypeMapping duplicate = new TypeMapping();
        duplicate.setMappingName(dtoMappingName.getMappingName());
        duplicate.setSourceDataStore(dtoMappingName.getSourceDataStore());
        duplicate.setSourceTypeName(dtoMappingName.getSourceTypeName());
        duplicate.setTargetElementName(dtoMappingName.getTargetElementName());
        mappings.add(dtoMappingName);
        mappings.add(duplicate);
        config.setTypeMappings(mappings);
        try {
            new AppSchemaDataAccess(AppSchemaDataAccessConfigurator.buildMappings(config));
        } catch (DataSourceException e) {
            assertTrue(e.getMessage()
                    .startsWith(
                            "Duplicate mappingName or targetElement across FeatureTypeMapping instances detected."));
            threwException = true;
        }
        assertTrue(threwException);
        threwException = false;
        /** Test when targetElement duplicates a mappingName */
        duplicate = new TypeMapping();
        duplicate.setMappingName(dtoNoMappingName.getTargetElementName());
        duplicate.setSourceDataStore(dtoNoMappingName.getSourceDataStore());
        duplicate.setSourceTypeName(dtoNoMappingName.getSourceTypeName());
        duplicate.setTargetElementName(dtoNoMappingName.getTargetElementName());
        mappings.clear();
        mappings.add(duplicate);
        mappings.add(dtoNoMappingName);
        config.setTypeMappings(mappings);
        // make sure the above operation didn't fail
        assertTrue(config.getTypeMappings().containsAll(mappings));
        try {
            new AppSchemaDataAccess(AppSchemaDataAccessConfigurator.buildMappings(config));
        } catch (DataSourceException e) {
            assertTrue(e.getMessage()
                    .startsWith(
                            "Duplicate mappingName or targetElement across FeatureTypeMapping instances detected."));
            threwException = true;
        }
        assertTrue(threwException);
        threwException = false;
        /** Test duplicate targetElement, when both don't have mappingName */
        duplicate = new TypeMapping();
        duplicate.setSourceDataStore(dtoNoMappingName.getSourceDataStore());
        duplicate.setSourceTypeName(dtoNoMappingName.getSourceTypeName());
        duplicate.setTargetElementName(dtoNoMappingName.getTargetElementName());
        mappings.clear();
        mappings.add(duplicate);
        mappings.add(dtoNoMappingName);
        config.setTypeMappings(mappings);
        assertTrue(config.getTypeMappings().containsAll(mappings));
        try {
            new AppSchemaDataAccess(AppSchemaDataAccessConfigurator.buildMappings(config));
        } catch (DataSourceException e) {
            assertTrue(e.getMessage()
                    .startsWith(
                            "Duplicate mappingName or targetElement across FeatureTypeMapping instances detected."));
            threwException = true;
        }
        assertTrue(threwException);
    }

    /** Success scenarios for keeping uniqueness of FeatureTypeMapping key (mappingName or targetElement). */
    @Test
    public void testUniqueKey() throws IOException {
        /** When mappingName are present in both mappings, and they're unique */
        Set<TypeMapping> mappings = new HashSet<>();
        TypeMapping duplicate = new TypeMapping();
        duplicate.setMappingName(dtoMappingName.getTargetElementName());
        duplicate.setSourceDataStore(dtoMappingName.getSourceDataStore());
        duplicate.setSourceTypeName(dtoMappingName.getSourceTypeName());
        duplicate.setTargetElementName(dtoMappingName.getTargetElementName());
        mappings.add(dtoMappingName);
        mappings.add(duplicate);
        config.setTypeMappings(mappings);
        AppSchemaDataAccess da = new AppSchemaDataAccess(AppSchemaDataAccessConfigurator.buildMappings(config));
        assertNotNull(da);
        da.dispose();
        /** When mappingName is present in one mapping, and it's different from the targetElement of the other. */
        mappings.clear();
        mappings.add(dtoMappingName);
        mappings.add(dtoNoMappingName);
        config.setTypeMappings(mappings);
        assertTrue(config.getTypeMappings().containsAll(mappings));
        da = new AppSchemaDataAccess(AppSchemaDataAccessConfigurator.buildMappings(config));
        assertNotNull(da);
        da.dispose();
        // no need to test the scenario if both target elements are unique, as most of the other
        // app-schema test files are already like this.
    }

    @Test
    public void testAppSchemaInRegistry() throws Exception {
        boolean threwException = false;
        Set<TypeMapping> mappings = new HashSet<>();
        mappings.add(dtoMappingName);
        config.setTypeMappings(mappings);
        AppSchemaDataAccess da = new AppSchemaDataAccess(AppSchemaDataAccessConfigurator.buildMappings(config));
        DataAccessRegistry.register(da);
        try {
            new AppSchemaDataAccess(AppSchemaDataAccessConfigurator.buildMappings(config, true));
        } catch (DataSourceException e) {
            assertTrue(e.getMessage()
                    .startsWith(
                            "Duplicate mappingName or targetElement across FeatureTypeMapping instances detected."));
            threwException = true;
        }
        // No exception because the mapping is from an include and matches the existing one
        assertFalse(threwException);
        try {
            new AppSchemaDataAccess(AppSchemaDataAccessConfigurator.buildMappings(config, false));
        } catch (DataSourceException e) {
            assertTrue(e.getMessage()
                    .startsWith(
                            "Duplicate mappingName or targetElement across FeatureTypeMapping instances detected."));
            threwException = true;
        }
        // Exception because the mapping is not from an include
        assertTrue(threwException);
        AttributeMapping attributeMapping = new AttributeMapping();
        attributeMapping.setIndexField("indexField");
        attributeMapping.setIdentifierPath("identifierPath");
        attributeMapping.setTargetAttributePath("targetAttributePath");
        mappings.iterator().next().getAttributeMappings().add(attributeMapping);
        try {
            new AppSchemaDataAccess(AppSchemaDataAccessConfigurator.buildMappings(config, true));
        } catch (DataSourceException e) {
            assertTrue(e.getMessage()
                    .startsWith(
                            "Duplicate mappingName or targetElement across FeatureTypeMapping instances detected."));
            threwException = true;
        }
        // Exception because the mapping is from an include but does not match the existing one
        assertTrue(threwException);
        DataAccessRegistry.unregister(da);
    }
}
