/*
 * Decompiled with CFR 0.152.
 */
package org.geoserver.gwc.layer;

import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;
import com.thoughtworks.xstream.XStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;
import java.util.Collection;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.geoserver.gwc.layer.GeoServerTileLayerInfo;
import org.geoserver.gwc.layer.GeoServerTileLayerInfoImpl;
import org.geoserver.gwc.layer.TileLayerCatalog;
import org.geoserver.platform.GeoServerResourceLoader;
import org.geotools.util.logging.Logging;
import org.geowebcache.config.XMLConfiguration;
import org.geowebcache.storage.blobstore.file.FilePathGenerator;

public class DefaultTileLayerCatalog
implements TileLayerCatalog {
    private static final Logger LOGGER = Logging.getLogger(DefaultTileLayerCatalog.class);
    private static final String LAYERINFO_DIRECTORY = "gwc-layers";
    private BiMap<String, String> layersById;
    private BiMap<String, String> layersByName;
    private final XStream serializer;
    private final GeoServerResourceLoader resourceLoader;
    private final String baseDirectory;
    private volatile boolean initialized;

    public DefaultTileLayerCatalog(GeoServerResourceLoader resourceLoader, XMLConfiguration xmlPersisterFactory) throws IOException {
        this(resourceLoader, xmlPersisterFactory.getConfiguredXStream(new XStream()));
    }

    DefaultTileLayerCatalog(GeoServerResourceLoader resourceLoader, XStream configuredXstream) throws IOException {
        this.resourceLoader = resourceLoader;
        this.serializer = configuredXstream;
        this.baseDirectory = LAYERINFO_DIRECTORY;
        HashBiMap baseBiMap = HashBiMap.create();
        this.layersById = Maps.synchronizedBiMap((BiMap)baseBiMap);
        this.layersByName = this.layersById.inverse();
        this.initialized = false;
    }

    @Override
    public void reset() {
        this.layersById.clear();
        this.initialized = false;
    }

    @Override
    public void initialize() {
        File baseDir;
        this.layersById.clear();
        try {
            baseDir = this.resourceLoader.findOrCreateDirectory(this.baseDirectory);
        }
        catch (IOException e) {
            throw Throwables.propagate((Throwable)e);
        }
        LOGGER.info("GeoServer TileLayer store base directory is: " + baseDir.getAbsolutePath());
        String[] tileLayerFiles = baseDir.list(new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return name.endsWith(".xml");
            }
        });
        LOGGER.info("Loading tile layers from " + baseDir.getAbsolutePath());
        for (String fileName : tileLayerFiles) {
            GeoServerTileLayerInfoImpl info;
            try {
                File file = new File(baseDir, fileName);
                info = this.depersist(file);
            }
            catch (Exception e) {
                LOGGER.log(Level.SEVERE, "Error depersisting tile layer information from file " + fileName, e);
                continue;
            }
            this.layersById.put((Object)info.getId(), (Object)info.getName());
            if (!LOGGER.isLoggable(Level.FINER)) continue;
            LOGGER.finer("Loaded tile layer '" + info.getName() + "'");
        }
        this.initialized = true;
    }

    @Override
    public GeoServerTileLayerInfo getLayerById(String id) {
        this.checkInitialized();
        if (!this.layersById.containsKey((Object)id)) {
            return null;
        }
        try {
            GeoServerTileLayerInfoImpl real = this.loadInternal(id);
            return real;
        }
        catch (IOException e) {
            LOGGER.finer("GeoServer tile layer does not exist or can't be loaded: " + id);
            LOGGER.log(Level.FINEST, "Trying to load tile layer " + id, e);
            return null;
        }
    }

    private synchronized void checkInitialized() {
        if (!this.initialized) {
            this.initialize();
        }
    }

    @Override
    public GeoServerTileLayerInfo getLayerByName(String layerName) {
        this.checkInitialized();
        String id = (String)this.layersByName.get((Object)layerName);
        if (id == null) {
            return null;
        }
        return this.getLayerById(id);
    }

    @Override
    public Set<String> getLayerIds() {
        this.checkInitialized();
        return ImmutableSet.copyOf((Collection)this.layersById.keySet());
    }

    @Override
    public boolean exists(String layerId) {
        this.checkInitialized();
        return this.layersById.containsKey((Object)layerId);
    }

    @Override
    public Set<String> getLayerNames() {
        this.checkInitialized();
        return ImmutableSet.copyOf((Collection)this.layersByName.keySet());
    }

    @Override
    public GeoServerTileLayerInfo delete(String tileLayerId) {
        this.checkInitialized();
        try {
            GeoServerTileLayerInfo info = this.getLayerById(tileLayerId);
            if (info != null) {
                File file = this.getFile(tileLayerId, false);
                this.layersById.remove((Object)tileLayerId);
                file.delete();
            }
            return info;
        }
        catch (IOException notFound) {
            LOGGER.log(Level.FINEST, "Deleting " + tileLayerId, notFound);
            return null;
        }
    }

    @Override
    public GeoServerTileLayerInfo save(GeoServerTileLayerInfo newValue) {
        this.checkInitialized();
        GeoServerTileLayerInfoImpl oldValue = null;
        String tileLayerId = newValue.getId();
        Preconditions.checkNotNull((Object)tileLayerId);
        try {
            try {
                oldValue = this.loadInternal(tileLayerId);
            }
            catch (FileNotFoundException ignore) {
            }
            catch (Exception other) {
                throw Throwables.propagate((Throwable)other);
            }
            if (oldValue == null) {
                String duplicateNameId = (String)this.layersByName.get((Object)newValue.getName());
                if (null != duplicateNameId) {
                    throw new IllegalArgumentException("TileLayer with same name already exists: " + newValue.getName() + ": <" + duplicateNameId + ">");
                }
            } else {
                this.layersByName.remove((Object)oldValue.getName());
            }
            this.persist(newValue);
            this.layersById.put((Object)newValue.getId(), (Object)newValue.getName());
        }
        catch (Exception e) {
            if (e instanceof ExecutionException) {
                Throwables.propagate((Throwable)((ExecutionException)e).getCause());
            }
            Throwables.propagate((Throwable)e);
        }
        return oldValue;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void persist(GeoServerTileLayerInfo real) throws IOException {
        String tileLayerId = real.getId();
        File file = this.getFile(tileLayerId, false);
        boolean cleanup = false;
        if (file == null) {
            cleanup = true;
            file = this.getFile(tileLayerId, true);
        }
        File tmp = new File(file.getParentFile(), file.getName() + ".tmp");
        try {
            OutputStreamWriter writer = new OutputStreamWriter((OutputStream)new FileOutputStream(tmp), "UTF-8");
            try {
                this.serializer.toXML((Object)real, (Writer)writer);
            }
            finally {
                ((Writer)writer).close();
            }
        }
        catch (Exception e) {
            tmp.delete();
            if (cleanup) {
                file.delete();
            }
            Throwables.propagateIfInstanceOf((Throwable)e, IOException.class);
            throw Throwables.propagate((Throwable)e);
        }
        try {
            this.depersist(tmp);
        }
        catch (Exception e) {
            LOGGER.log(Level.WARNING, "Persisted version of tile layer " + real.getName() + " can't be loaded back", e);
            Throwables.propagateIfInstanceOf((Throwable)e, IOException.class);
            throw Throwables.propagate((Throwable)e);
        }
        this.rename(tmp, file);
    }

    private GeoServerTileLayerInfoImpl loadInternal(String tileLayerId) throws FileNotFoundException, IOException {
        File file = this.getFile(tileLayerId, false);
        if (null == file) {
            throw new FileNotFoundException(tileLayerId);
        }
        return this.depersist(file);
    }

    private File getFile(String tileLayerId, boolean create) throws IOException {
        String fileName = FilePathGenerator.filteredLayerName((String)tileLayerId) + ".xml";
        File base = this.resourceLoader.findOrCreateDirectory(this.baseDirectory);
        File file = this.resourceLoader.find(base, fileName);
        if (null == file && create) {
            return this.resourceLoader.createFile(base, fileName);
        }
        return file;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private GeoServerTileLayerInfoImpl depersist(File file) throws IOException {
        GeoServerTileLayerInfoImpl info;
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.fine("Depersisting GeoServerTileLayerInfo from " + file.getAbsolutePath());
        }
        InputStreamReader reader = new InputStreamReader((InputStream)new FileInputStream(file), "UTF-8");
        try {
            info = (GeoServerTileLayerInfoImpl)this.serializer.fromXML((Reader)reader);
        }
        finally {
            ((Reader)reader).close();
        }
        return info;
    }

    private void rename(File source, File dest) throws IOException {
        if (source.getCanonicalPath().equalsIgnoreCase(dest.getCanonicalPath())) {
            return;
        }
        boolean win = System.getProperty("os.name").startsWith("Windows");
        if (win && dest.exists()) {
            if (!dest.delete()) {
                throw new IOException("Could not delete: " + dest.getCanonicalPath());
            }
            source.renameTo(dest);
        } else {
            source.renameTo(dest);
        }
    }

    @Override
    public String getLayerId(String layerName) {
        this.checkInitialized();
        return (String)this.layersByName.get((Object)layerName);
    }

    @Override
    public String getLayerName(String layerId) {
        this.checkInitialized();
        return (String)this.layersById.get((Object)layerId);
    }
}

