/* (c) 2014 Open Source Geospatial Foundation - all rights reserved
 * This code is licensed under the GPL 2.0 license, available at the root
 * application directory.
 */
package org.geoserver.filters;

import java.io.IOException;
import java.util.zip.GZIPOutputStream;
import javax.servlet.ServletOutputStream;
import javax.servlet.WriteListener;
import javax.servlet.http.HttpServletResponse;

/**
 * A simple streaming gzipping servlet output stream wrapper
 *
 * @author Andrea Aime - GeoSolutions
 */
public class GZIPResponseStream extends ServletOutputStream {
    protected final ServletOutputStream delegateStream;
    protected GZIPOutputStream gzipstream = null;

    protected boolean closed = false;

    public GZIPResponseStream(HttpServletResponse response) throws IOException {
        super();
        closed = false;
        delegateStream = response.getOutputStream();
        gzipstream = new GZIPOutputStream(delegateStream, 4096, true);
    }

    @Override
    public void close() throws IOException {
        if (closed) {
            throw new IOException("This output stream has already been closed");
        }
        gzipstream.finish();
        closed = true;
    }

    @Override
    public void flush() throws IOException {
        if (!closed) {
            gzipstream.flush();
        }
    }

    @Override
    public void write(int b) throws IOException {
        if (closed) {
            throw new IOException("Cannot write to a closed output stream");
        }
        gzipstream.write((byte) b);
    }

    @Override
    public void write(byte[] b) throws IOException {
        write(b, 0, b.length);
    }

    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        if (closed) {
            throw new IOException("Cannot write to a closed output stream");
        }
        gzipstream.write(b, off, len);
    }

    public boolean closed() {
        return (this.closed);
    }

    @Override
    public boolean isReady() {
        return delegateStream.isReady();
    }

    @Override
    public void setWriteListener(WriteListener writeListener) {
        delegateStream.setWriteListener(writeListener);
    }
}
