/*
 * Decompiled with CFR 0.152.
 */
package org.geoserver.geofence.server.integration;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.geoserver.catalog.Catalog;
import org.geoserver.catalog.CatalogBuilder;
import org.geoserver.catalog.LayerGroupInfo;
import org.geoserver.catalog.LayerInfo;
import org.geoserver.catalog.WorkspaceInfo;
import org.geoserver.data.test.MockData;
import org.geoserver.data.test.SystemTestData;
import org.geoserver.geofence.GeofenceAccessManager;
import org.geoserver.geofence.config.GeoFenceConfiguration;
import org.geoserver.geofence.config.GeoFenceConfigurationManager;
import org.geoserver.geofence.core.model.enums.AdminGrantType;
import org.geoserver.geofence.core.model.enums.CatalogMode;
import org.geoserver.geofence.core.model.enums.GrantType;
import org.geoserver.geofence.core.model.enums.SpatialFilterType;
import org.geoserver.geofence.server.integration.GeofenceIntegrationTestSupport;
import org.geoserver.geofence.services.RuleAdminService;
import org.geoserver.ows.Dispatcher;
import org.geoserver.ows.Request;
import org.geoserver.security.VectorAccessLimits;
import org.geoserver.security.WorkspaceAccessLimits;
import org.geoserver.test.GeoServerSystemTestSupport;
import org.geotools.api.filter.Filter;
import org.geotools.api.filter.spatial.Intersects;
import org.geotools.api.referencing.FactoryException;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;
import org.geotools.api.referencing.operation.MathTransform;
import org.geotools.api.referencing.operation.TransformException;
import org.geotools.geometry.jts.JTS;
import org.geotools.geometry.jts.ReferencedEnvelope;
import org.geotools.referencing.CRS;
import org.geotools.util.Converters;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.locationtech.jts.geom.Envelope;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.MultiPolygon;
import org.locationtech.jts.io.ParseException;
import org.locationtech.jts.io.WKTReader;
import org.locationtech.jts.io.WKTWriter;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;

public class GeofenceAccessManagerIntegrationTest
extends GeoServerSystemTestSupport {
    public GeofenceIntegrationTestSupport support;
    private GeofenceAccessManager accessManager;
    private GeoFenceConfigurationManager configManager;
    private RuleAdminService ruleService;
    private static final String AREA_WKT = "MULTIPOLYGON(((0.0016139656066815888 -0.0006386457758059581,0.0019599705696027314 -0.0006386457758059581,0.0019599705696027314 -0.0008854090051601674,0.0016139656066815888 -0.0008854090051601674,0.0016139656066815888 -0.0006386457758059581)))";
    private static final String AREA_WKT_2 = "MULTIPOLYGON(((0.0011204391479413545 -0.0006405065746780663,0.0015764146804730927 -0.0006405065746780663,0.0015764146804730927 -0.0014612625330857614,0.0011204391479413545 -0.0014612625330857614,0.0011204391479413545 -0.0006405065746780663)))";
    private static final String AREA_WKT_3 = "MULTIPOLYGON (((0.00136827777777778 0.002309, 0.00372027777777778 0.00224366666666667, 0.00244083333333333 -0.00133877777777778, 0.00044272222222222 -0.00131155555555556, 0.00136827777777778 0.002309)))";
    private static final String AREA_WKT_4 = "MULTIPOLYGON (((0.00099261111111111 0.00175366666666667, 0.00298527777777778 0.00110577777777778, 0.00188005555555556 -0.00123533333333333, 0.00107972222222222 -0.00126255555555556, 0.00057338888888889 0.00096422222222222, 0.00099261111111111 0.00175366666666667)))";
    private static final String AREA_WKT_INTERSECT_1 = "MULTIPOLYGON (((-0.15605493133583015 0.52434456928838946, 0.22097378277153568 0.51435705368289641, 0.22846441947565554 0.2247191011235955, -0.06866416978776524 0.23470661672908866, -0.15605493133583015 0.52434456928838946)))";
    private static final String AREA_WKT_INTERSECT_2 = "MULTIPOLYGON (((-0.2359550561797753 0.36704119850187267, 0.37328339575530589 0.33957553058676659, 0.37328339575530589 0.25468164794007497, -0.16104868913857673 0.25717852684144826, -0.2359550561797753 0.36704119850187267)))";

    @Before
    public void setUp() {
        this.support = new GeofenceIntegrationTestSupport(() -> GeoServerSystemTestSupport.applicationContext);
        this.support.before();
        this.accessManager = (GeofenceAccessManager)applicationContext.getBean("geofenceRuleAccessManager", GeofenceAccessManager.class);
        this.configManager = (GeoFenceConfigurationManager)applicationContext.getBean(GeoFenceConfigurationManager.class);
        this.ruleService = (RuleAdminService)applicationContext.getBean("ruleAdminService");
        if (this.ruleService.getRuleByPriority(9999L) == null) {
            this.support.addRule(GrantType.ALLOW, null, null, null, null, null, null, 9999L);
        }
    }

    @After
    public void clearRules() {
        this.support.after();
        SecurityContextHolder.getContext().setAuthentication(null);
    }

    protected void setUpTestData(SystemTestData testData) throws Exception {
        super.setUpTestData(testData);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testAllowedAreaLayerInTwoGroupsEnlargement() throws Exception {
        Long idRule = null;
        LayerGroupInfo group1 = null;
        LayerGroupInfo group2 = null;
        try {
            Authentication user = this.getUser("anonymousUser", "", "ROLE_ANONYMOUS");
            this.login("admin", "geoserver", new String[]{"ROLE_ADMINISTRATOR"});
            Catalog catalog = this.getCatalog();
            LayerInfo places = catalog.getLayerByName(this.getLayerId(MockData.NAMED_PLACES));
            LayerInfo forests = catalog.getLayerByName(this.getLayerId(MockData.FORESTS));
            group1 = this.createsLayerGroup("group21", LayerGroupInfo.Mode.NAMED, null, Arrays.asList(places, forests));
            group2 = this.createsLayerGroup("group22", LayerGroupInfo.Mode.NAMED, null, Arrays.asList(places, forests));
            idRule = this.support.addRule(GrantType.LIMIT, "anonymousUser", "ROLE_ANONYMOUS", "WMS", null, null, "group21", 0L);
            this.support.addRule(GrantType.LIMIT, "anonymousUser", "ROLE_ANONYMOUS", "WMS", null, null, "group22", 1L);
            this.support.addRuleLimits(idRule, CatalogMode.HIDE, AREA_WKT, 4326);
            Request req = new Request();
            req.setService("WMS");
            req.setRequest("GetMap");
            Dispatcher.REQUEST.set(req);
            this.logout();
            this.login("anonymousUser", "", new String[]{"ROLE_ANONYMOUS"});
            VectorAccessLimits vl = (VectorAccessLimits)this.accessManager.getAccessLimits(user, places);
            Assert.assertEquals((Object)vl.getReadFilter(), (Object)Filter.INCLUDE);
            Assert.assertEquals((Object)vl.getWriteFilter(), (Object)Filter.INCLUDE);
            this.logout();
        }
        catch (Throwable throwable) {
            this.removeLayerGroup(group1, group2);
            throw throwable;
        }
        this.removeLayerGroup(group1, group2);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testAllowedAreaLayerInTwoGroupsRestrictAccess() throws Exception {
        Long idRule = null;
        Long idRule2 = null;
        LayerGroupInfo group1 = null;
        LayerGroupInfo group2 = null;
        try {
            Authentication user = this.getUser("anonymousUser", "", "ROLE_ANONYMOUS");
            this.login("admin", "geoserver", new String[]{"ROLE_ADMINISTRATOR"});
            Catalog catalog = this.getCatalog();
            LayerInfo bridges = catalog.getLayerByName(this.getLayerId(MockData.BRIDGES));
            LayerInfo buildings = catalog.getLayerByName(this.getLayerId(MockData.BUILDINGS));
            group1 = this.createsLayerGroup("group1", LayerGroupInfo.Mode.NAMED, null, Arrays.asList(bridges, buildings));
            group2 = this.createsLayerGroup("group2", LayerGroupInfo.Mode.NAMED, null, Arrays.asList(bridges, buildings));
            idRule = this.support.addRule(GrantType.LIMIT, "anonymousUser", "ROLE_ANONYMOUS", "WMS", null, null, "group1", 2L);
            this.support.addRuleLimits(idRule, CatalogMode.HIDE, AREA_WKT_INTERSECT_1, 4326);
            idRule2 = this.support.addRule(GrantType.LIMIT, "anonymousUser", "ROLE_ANONYMOUS", "WMS", null, null, "group2", 3L);
            this.support.addRuleLimits(idRule2, CatalogMode.HIDE, AREA_WKT_INTERSECT_2, 4326);
            Request req = new Request();
            req.setService("WMS");
            req.setRequest("GetMap");
            Dispatcher.REQUEST.set(req);
            this.logout();
            this.login("anonymousUser", "", new String[]{"ROLE_ANONYMOUS"});
            VectorAccessLimits vl = (VectorAccessLimits)this.accessManager.getAccessLimits(user, bridges);
            Geometry allowedArea1 = new WKTReader().read(AREA_WKT_INTERSECT_1);
            Geometry allowedArea2 = new WKTReader().read(AREA_WKT_INTERSECT_2);
            MultiPolygon intersectionArea = (MultiPolygon)Converters.convert((Object)allowedArea1.intersection(allowedArea2), MultiPolygon.class);
            Intersects intersects = (Intersects)vl.getReadFilter();
            MultiPolygon readFilterArea = (MultiPolygon)intersects.getExpression2().evaluate(null, MultiPolygon.class);
            Intersects intersects2 = (Intersects)vl.getWriteFilter();
            MultiPolygon writeFilterArea = (MultiPolygon)intersects2.getExpression2().evaluate(null, MultiPolygon.class);
            intersectionArea.normalize();
            readFilterArea.normalize();
            writeFilterArea.normalize();
            Assert.assertTrue((boolean)intersectionArea.equalsExact((Geometry)readFilterArea, 1.0E-14));
            Assert.assertTrue((boolean)intersectionArea.equalsExact((Geometry)writeFilterArea, 1.0E-14));
            this.logout();
        }
        catch (Throwable throwable) {
            this.removeLayerGroup(group1, group2);
            throw throwable;
        }
        this.removeLayerGroup(group1, group2);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testAllowedAreaLayerInTwoGroupsEnlargement2() throws Exception {
        LayerGroupInfo group1 = null;
        LayerGroupInfo group2 = null;
        try {
            Authentication user = this.getUser("anonymousUser", "", "ROLE_ANONYMOUS", "ROLE_ANONYMOUS2");
            this.login("admin", "geoserver", new String[]{"ROLE_ADMINISTRATOR"});
            Catalog catalog = this.getCatalog();
            LayerInfo lakes = catalog.getLayerByName(this.getLayerId(MockData.LAKES));
            LayerInfo fifteen = catalog.getLayerByName(this.getLayerId(MockData.FIFTEEN));
            group1 = this.createsLayerGroup("group31", LayerGroupInfo.Mode.NAMED, null, Arrays.asList(lakes, fifteen));
            group2 = this.createsLayerGroup("group32", LayerGroupInfo.Mode.NAMED, null, Arrays.asList(lakes, fifteen));
            Long idRule = this.support.addRule(GrantType.LIMIT, "anonymousUser", "ROLE_ANONYMOUS", "WMS", null, null, "group31", 4L);
            Long idRule2 = this.support.addRule(GrantType.LIMIT, "anonymousUser", "ROLE_ANONYMOUS2", "WMS", null, null, "group32", 5L);
            Long idRule3 = this.support.addRule(GrantType.LIMIT, "anonymousUser", "ROLE_ANONYMOUS", "WMS", null, null, "group32", 4L);
            Long idRule4 = this.support.addRule(GrantType.LIMIT, "anonymousUser", "ROLE_ANONYMOUS2", "WMS", null, null, "group31", 5L);
            this.support.addRuleLimits(idRule, CatalogMode.HIDE, AREA_WKT, 4326);
            this.support.addRuleLimits(idRule2, CatalogMode.HIDE, AREA_WKT_2, 4326);
            this.support.addRuleLimits(idRule3, CatalogMode.HIDE, AREA_WKT_3, 4326);
            this.support.addRuleLimits(idRule4, CatalogMode.HIDE, AREA_WKT_3, 4326);
            Geometry allowedArea1 = new WKTReader().read(AREA_WKT);
            Geometry allowedArea2 = new WKTReader().read(AREA_WKT_2);
            Geometry allowedArea3 = new WKTReader().read(AREA_WKT_3);
            Geometry intersectOne = allowedArea1.intersection(allowedArea3);
            Geometry intersectTwo = allowedArea2.intersection(allowedArea3);
            MultiPolygon unionedArea = (MultiPolygon)intersectOne.union(intersectTwo);
            unionedArea.normalize();
            Request req = new Request();
            req.setService("WMS");
            req.setRequest("GetMap");
            Dispatcher.REQUEST.set(req);
            this.logout();
            this.login("anonymousUser", "", new String[]{"ROLE_ANONYMOUS", "ROLE_ANONYMOUS2"});
            VectorAccessLimits vl = (VectorAccessLimits)this.accessManager.getAccessLimits(user, lakes);
            Intersects readFilter = (Intersects)vl.getReadFilter();
            MultiPolygon readFilterArea = (MultiPolygon)readFilter.getExpression2().evaluate(null, MultiPolygon.class);
            readFilterArea.normalize();
            Intersects writeFilter = (Intersects)vl.getWriteFilter();
            MultiPolygon writeFilterArea = (MultiPolygon)writeFilter.getExpression2().evaluate(null, MultiPolygon.class);
            writeFilterArea.normalize();
            Assert.assertTrue((boolean)unionedArea.equalsExact((Geometry)readFilterArea, 1.0E-14));
            Assert.assertTrue((boolean)unionedArea.equalsExact((Geometry)writeFilterArea, 1.0E-14));
            this.logout();
        }
        catch (Throwable throwable) {
            this.removeLayerGroup(group1, group2);
            throw throwable;
        }
        this.removeLayerGroup(group1, group2);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testAllowedAreaSRIDIsPreserved() throws Exception {
        Long idRule = null;
        LayerGroupInfo group1 = null;
        try {
            Authentication user = this.getUser("anonymousUser", "", "ROLE_ANONYMOUS");
            this.login("admin", "geoserver", new String[]{"ROLE_ADMINISTRATOR"});
            Catalog catalog = this.getCatalog();
            LayerInfo basicPolygons = catalog.getLayerByName(this.getLayerId(MockData.BASIC_POLYGONS));
            LayerInfo fifteen = catalog.getLayerByName(this.getLayerId(MockData.FIFTEEN));
            group1 = this.createsLayerGroup("group41", LayerGroupInfo.Mode.NAMED, null, Arrays.asList(basicPolygons, fifteen));
            idRule = this.support.addRule(GrantType.LIMIT, "anonymousUser", "ROLE_ANONYMOUS", "WMS", null, null, "group41", 7L);
            this.support.addRuleLimits(idRule, CatalogMode.HIDE, AREA_WKT, 3857);
            Request req = new Request();
            req.setService("WMS");
            req.setRequest("GetMap");
            Dispatcher.REQUEST.set(req);
            this.logout();
            this.login("anonymousUser", "", new String[]{"ROLE_ANONYMOUS"});
            VectorAccessLimits vl = (VectorAccessLimits)this.accessManager.getAccessLimits(user, basicPolygons);
            Intersects intersects = (Intersects)vl.getReadFilter();
            MultiPolygon allowedArea = (MultiPolygon)intersects.getExpression2().evaluate(null, MultiPolygon.class);
            allowedArea.normalize();
            Geometry geom = new WKTReader().read(AREA_WKT);
            geom.setSRID(3857);
            MathTransform mt = CRS.findMathTransform((CoordinateReferenceSystem)CRS.decode((String)"EPSG:3857"), (CoordinateReferenceSystem)basicPolygons.getResource().getCRS(), (boolean)true);
            MultiPolygon reproj = (MultiPolygon)JTS.transform((Geometry)geom, (MathTransform)mt);
            reproj.normalize();
            Assert.assertTrue((boolean)allowedArea.equalsExact((Geometry)reproj, 1.0E-14));
            this.logout();
        }
        catch (Throwable throwable) {
            this.removeLayerGroup(group1);
            throw throwable;
        }
        this.removeLayerGroup(group1);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testLayerGroupsAllowedAreaWithDifferentSRIDS() throws Exception {
        LayerGroupInfo group1 = null;
        LayerGroupInfo group2 = null;
        try {
            Authentication user = this.getUser("anonymousUser", "", "ROLE_ANONYMOUS", "ROLE_ANONYMOUS2");
            this.login("admin", "geoserver", new String[]{"ROLE_ADMINISTRATOR"});
            Catalog catalog = this.getCatalog();
            LayerInfo lakes = catalog.getLayerByName(this.getLayerId(MockData.LAKES));
            LayerInfo namedPlaces = catalog.getLayerByName(this.getLayerId(MockData.NAMED_PLACES));
            group1 = this.createsLayerGroup("group51", LayerGroupInfo.Mode.NAMED, null, Arrays.asList(lakes, namedPlaces));
            group2 = this.createsLayerGroup("group52", LayerGroupInfo.Mode.NAMED, null, Arrays.asList(lakes, namedPlaces));
            Long idRule = this.support.addRule(GrantType.LIMIT, "anonymousUser", "ROLE_ANONYMOUS", "WMS", null, null, "group51", 8L);
            Long idRule2 = this.support.addRule(GrantType.LIMIT, "anonymousUser", "ROLE_ANONYMOUS2", "WMS", null, null, "group52", 9L);
            Long idRule3 = this.support.addRule(GrantType.LIMIT, "anonymousUser", "ROLE_ANONYMOUS", "WMS", null, null, "group52", 8L);
            Long idRule4 = this.support.addRule(GrantType.LIMIT, "anonymousUser", "ROLE_ANONYMOUS2", "WMS", null, null, "group51", 9L);
            this.support.addRuleLimits(idRule, CatalogMode.HIDE, AREA_WKT, 4326);
            this.support.addRuleLimits(idRule2, CatalogMode.HIDE, this.repWkt(AREA_WKT_2, CRS.decode((String)"EPSG:4326"), CRS.decode((String)"EPSG:3857"), 3857), 3857);
            this.support.addRuleLimits(idRule3, CatalogMode.HIDE, AREA_WKT_3, 4326);
            this.support.addRuleLimits(idRule4, CatalogMode.HIDE, AREA_WKT_4, 4326);
            Request req = new Request();
            req.setService("WMS");
            req.setRequest("GetMap");
            Dispatcher.REQUEST.set(req);
            this.logout();
            this.login("anonymousUser", "", new String[]{"ROLE_ANONYMOUS", "ROLE_ANONYMOUS2"});
            VectorAccessLimits vl = (VectorAccessLimits)this.accessManager.getAccessLimits(user, lakes);
            Intersects intersects = (Intersects)vl.getReadFilter();
            MultiPolygon allowedArea = (MultiPolygon)intersects.getExpression2().evaluate(null, MultiPolygon.class);
            allowedArea.normalize();
            Geometry geom = new WKTReader().read(AREA_WKT);
            geom.setSRID(4326);
            Geometry geom2 = new WKTReader().read(AREA_WKT_2);
            geom2.setSRID(4326);
            Geometry geom3 = new WKTReader().read(AREA_WKT_3);
            geom.setSRID(4326);
            Geometry geom4 = new WKTReader().read(AREA_WKT_4);
            geom.setSRID(4326);
            Geometry intersect1 = geom.intersection(geom3);
            Geometry intersect2 = geom2.intersection(geom4);
            Geometry union = intersect1.union(intersect2);
            union.setSRID(4326);
            union.normalize();
            Assert.assertTrue((boolean)allowedArea.equalsExact(union, 1.0E-11));
            this.logout();
        }
        catch (Throwable throwable) {
            this.removeLayerGroup(group1, group2);
            throw throwable;
        }
        this.removeLayerGroup(group1, group2);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testLayerGroupsClipAndIntersectsSpatialFilterUnion() throws Exception {
        Long idRule = null;
        Long idRule2 = null;
        Long idRule3 = null;
        Long idRule4 = null;
        LayerGroupInfo group1 = null;
        LayerGroupInfo group2 = null;
        try {
            Authentication user = this.getUser("anonymousUser", "", "ROLE_ANONYMOUS", "ROLE_ANONYMOUS2");
            this.login("admin", "geoserver", new String[]{"ROLE_ADMINISTRATOR"});
            Catalog catalog = this.getCatalog();
            LayerInfo droutes = catalog.getLayerByName(this.getLayerId(MockData.DIVIDED_ROUTES));
            LayerInfo ponds = catalog.getLayerByName(this.getLayerId(MockData.PONDS));
            group1 = this.createsLayerGroup("group61", LayerGroupInfo.Mode.NAMED, null, Arrays.asList(droutes, ponds));
            idRule = this.support.addRule(GrantType.LIMIT, "anonymousUser", "ROLE_ANONYMOUS", "WMS", null, null, "group61", 10L);
            idRule2 = this.support.addRule(GrantType.LIMIT, "anonymousUser", "ROLE_ANONYMOUS2", "WMS", null, null, "group61", 11L);
            group2 = this.createsLayerGroup("group62", LayerGroupInfo.Mode.NAMED, null, Arrays.asList(droutes, ponds));
            idRule3 = this.support.addRule(GrantType.LIMIT, "anonymousUser", "ROLE_ANONYMOUS", "WMS", null, null, "group62", 12L);
            idRule4 = this.support.addRule(GrantType.LIMIT, "anonymousUser", "ROLE_ANONYMOUS2", "WMS", null, null, "group62", 13L);
            String areaWKT1 = "MultiPolygon (((-97.48185823120911664 0.02172899055096349, -97.4667765271758384 0.02148629646307176, -97.46795532703131926 0.01663241470523705, -97.48165020770520073 0.01607768536148451, -97.48185823120911664 0.02172899055096349)))";
            String areaWKT2 = "MultiPolygon (((-97.48109547836145339 0.026374848804891, -97.46934215039070182 0.02672155464473634, -97.46993155031843514 0.02294246099042217, -97.48102613719348142 0.02294246099042217, -97.48109547836145339 0.026374848804891)))";
            String areaWKT3 = "MultiPolygon (((-97.48119949011341134 0.00914356856457779, -97.46941149155865958 0.00973296849231486, -97.46955017389460352 0.00605788658995429, -97.48182356062513065 0.00581519250206256, -97.48119949011341134 0.00914356856457779)))";
            String areaWKT4 = "MultiPolygon (((-97.48161553712121474 0.00449771031065027, -97.46889143279889822 0.00435902797471214, -97.46948083272663155 0.00127334600008865, -97.48178889004114467 0.00120400483211958, -97.48161553712121474 0.00449771031065027)))";
            this.support.addRuleLimits(idRule, CatalogMode.HIDE, areaWKT1, 4326);
            this.support.addRuleLimits(idRule2, CatalogMode.HIDE, areaWKT2, 4326, SpatialFilterType.CLIP);
            this.support.addRuleLimits(idRule3, CatalogMode.HIDE, areaWKT3, 4326);
            this.support.addRuleLimits(idRule4, CatalogMode.HIDE, areaWKT4, 4326, SpatialFilterType.CLIP);
            Request req = new Request();
            req.setService("WMS");
            req.setRequest("GetMap");
            Dispatcher.REQUEST.set(req);
            this.logout();
            this.login("anonymousUser", "", new String[]{"ROLE_ANONYMOUS", "ROLE_ANONYMOUS2"});
            VectorAccessLimits vl = (VectorAccessLimits)this.accessManager.getAccessLimits(user, droutes);
            Geometry intersectsArea = vl.getIntersectVectorFilter();
            Geometry clipArea = vl.getClipVectorFilter();
            intersectsArea.normalize();
            clipArea.normalize();
            WKTReader reader = new WKTReader();
            Geometry geom = reader.read(areaWKT1);
            geom.setSRID(4326);
            Geometry geom2 = new WKTReader().read(areaWKT2);
            geom2.setSRID(4326);
            Geometry geom3 = reader.read(areaWKT3);
            geom.setSRID(4326);
            Geometry geom4 = new WKTReader().read(areaWKT4);
            geom2.setSRID(4326);
            Geometry intersectIntersection = geom.intersection(geom3);
            intersectIntersection.setSRID(4326);
            intersectIntersection.normalize();
            Geometry clipIntersection = geom2.intersection(geom4);
            clipIntersection.setSRID(4326);
            clipIntersection.normalize();
            Assert.assertTrue((boolean)intersectsArea.equalsExact(intersectIntersection, 1.0E-14));
            Assert.assertTrue((boolean)clipArea.equalsExact(clipIntersection, 1.0E-14));
            this.logout();
        }
        catch (Throwable throwable) {
            this.removeLayerGroup(group1, group2);
            throw throwable;
        }
        this.removeLayerGroup(group1, group2);
    }

    @Test
    public void testCiteCannotWriteOnWorkspace() {
        this.configManager.getConfiguration().setGrantWriteToWorkspacesToAuthenticatedUsers(false);
        Authentication user = this.getUser("cite", "cite", "ROLE_AUTHENTICATED");
        WorkspaceInfo citeWS = this.getCatalog().getWorkspaceByName(MockData.CITE_PREFIX);
        WorkspaceAccessLimits wl = this.accessManager.getAccessLimits(user, citeWS);
        Assert.assertTrue((boolean)wl.isReadable());
        Assert.assertFalse((boolean)wl.isWritable());
    }

    @Test
    public void testAdminRule_WorkspaceAccessLimits_Role_based_rule() {
        String citeUserRole = "CITE_USER";
        String citeAdminRole = "CITE_ADMIN";
        String sfUserRole = "SF_USER";
        String sfAdminRole = "SF_ADMIN";
        Authentication citeUser = this.getUser("citeuser", "cite", "CITE_USER");
        Authentication citeAdmin = this.getUser("citeadmin", "cite", "CITE_ADMIN");
        Authentication sfUser = this.getUser("sfuser", "sfuser", "SF_USER");
        Authentication sfAdmin = this.getUser("sfadmin", "sfadmin", "SF_ADMIN");
        WorkspaceInfo cite = this.getCatalog().getWorkspaceByName("cite");
        WorkspaceInfo sf = this.getCatalog().getWorkspaceByName("sf");
        this.support.addAdminRule(0L, null, "CITE_ADMIN", cite.getName(), AdminGrantType.ADMIN);
        this.support.addAdminRule(1L, null, "CITE_USER", cite.getName(), AdminGrantType.USER);
        this.support.addAdminRule(2L, null, "SF_ADMIN", sf.getName(), AdminGrantType.ADMIN);
        this.support.addAdminRule(3L, null, "SF_USER", sf.getName(), AdminGrantType.USER);
        this.setUser(citeUser);
        this.assertAdminAccess(citeUser, cite, false);
        this.assertAdminAccess(citeUser, sf, false);
        this.setUser(sfUser);
        this.assertAdminAccess(sfUser, cite, false);
        this.assertAdminAccess(sfUser, sf, false);
        this.setUser(citeAdmin);
        this.assertAdminAccess(citeAdmin, cite, true);
        this.assertAdminAccess(citeAdmin, sf, false);
        this.setUser(sfAdmin);
        this.assertAdminAccess(sfAdmin, cite, false);
        this.assertAdminAccess(sfAdmin, sf, true);
    }

    @Test
    public void testAdminRule_WorkspaceAccessLimits_Username_based_rule() {
        String citeUserRole = "CITE_USER";
        String citeAdminRole = "CITE_ADMIN";
        String sfUserRole = "SF_USER";
        String sfAdminRole = "SF_ADMIN";
        Authentication citeUser = this.getUser("citeuser", "cite", "CITE_USER");
        Authentication citeAdmin = this.getUser("citeadmin", "cite", "CITE_ADMIN");
        Authentication sfUser = this.getUser("sfuser", "sfuser", "SF_USER");
        Authentication sfAdmin = this.getUser("sfadmin", "sfadmin", "SF_ADMIN");
        WorkspaceInfo cite = this.getCatalog().getWorkspaceByName("cite");
        WorkspaceInfo sf = this.getCatalog().getWorkspaceByName("sf");
        this.support.addAdminRule(0L, citeAdmin.getName(), null, cite.getName(), AdminGrantType.ADMIN);
        this.support.addAdminRule(1L, citeUser.getName(), null, cite.getName(), AdminGrantType.USER);
        this.support.addAdminRule(2L, sfAdmin.getName(), null, sf.getName(), AdminGrantType.ADMIN);
        this.support.addAdminRule(3L, sfUser.getName(), null, sf.getName(), AdminGrantType.USER);
        this.assertAdminAccess(citeUser, cite, false);
        this.assertAdminAccess(citeUser, sf, false);
        this.assertAdminAccess(sfUser, cite, false);
        this.assertAdminAccess(sfUser, sf, false);
        this.assertAdminAccess(citeAdmin, cite, true);
        this.assertAdminAccess(citeAdmin, sf, false);
        this.assertAdminAccess(sfAdmin, cite, false);
        this.assertAdminAccess(sfAdmin, sf, true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testAllowedAreaLayerInTwoGroupsEnlargementWithSingle() throws Exception {
        Long idRule = null;
        LayerGroupInfo group1 = null;
        LayerGroupInfo group2 = null;
        try {
            Authentication user = this.getUser("anonymousUser", "", "ROLE_ANONYMOUS");
            this.login("admin", "geoserver", new String[]{"ROLE_ADMINISTRATOR"});
            Catalog catalog = this.getCatalog();
            LayerInfo places = catalog.getLayerByName(this.getLayerId(MockData.NAMED_PLACES));
            LayerInfo forests = catalog.getLayerByName(this.getLayerId(MockData.FORESTS));
            group1 = this.createsLayerGroup("group21", LayerGroupInfo.Mode.SINGLE, null, Arrays.asList(places, forests));
            group2 = this.createsLayerGroup("group22", LayerGroupInfo.Mode.NAMED, null, Arrays.asList(places, forests));
            idRule = this.support.addRule(GrantType.LIMIT, "anonymousUser", "ROLE_ANONYMOUS", "WMS", null, null, "group21", 0L);
            this.support.addRule(GrantType.LIMIT, "anonymousUser", "ROLE_ANONYMOUS", "WMS", null, null, "group22", 1L);
            this.support.addRuleLimits(idRule, CatalogMode.HIDE, AREA_WKT, 4326);
            Request req = new Request();
            req.setService("WMS");
            req.setRequest("GetMap");
            Dispatcher.REQUEST.set(req);
            this.logout();
            this.login("anonymousUser", "", new String[]{"ROLE_ANONYMOUS"});
            VectorAccessLimits vl = (VectorAccessLimits)this.accessManager.getAccessLimits(user, places);
            Assert.assertEquals((Object)vl.getReadFilter(), (Object)Filter.INCLUDE);
            Assert.assertEquals((Object)vl.getWriteFilter(), (Object)Filter.INCLUDE);
            this.logout();
        }
        catch (Throwable throwable) {
            this.removeLayerGroup(group1, group2);
            throw throwable;
        }
        this.removeLayerGroup(group1, group2);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testLayerInGroupAreaEnlargement() throws Exception {
        Long idRule = null;
        Long idRule2 = null;
        LayerGroupInfo group1 = null;
        try {
            Authentication user = this.getUser("anonymousUser", "", "ROLE_ANONYMOUS", "ROLE_ANONYMOUS2");
            this.login("admin", "geoserver", new String[]{"ROLE_ADMINISTRATOR"});
            Catalog catalog = this.getCatalog();
            LayerInfo lakes = catalog.getLayerByName(this.getLayerId(MockData.LAKES));
            LayerInfo fifteen = catalog.getLayerByName(this.getLayerId(MockData.FIFTEEN));
            group1 = this.createsLayerGroup("group31", LayerGroupInfo.Mode.NAMED, null, Arrays.asList(lakes, fifteen));
            idRule = this.support.addRule(GrantType.LIMIT, null, "ROLE_ANONYMOUS", "WMS", null, null, "group31", 3L);
            idRule2 = this.support.addRule(GrantType.LIMIT, null, "ROLE_ANONYMOUS2", "WMS", null, "cite", "Lakes", 2L);
            this.support.addRuleLimits(idRule, CatalogMode.HIDE, AREA_WKT, 4326);
            this.support.addRuleLimits(idRule2, CatalogMode.HIDE, AREA_WKT_2, 4326);
            Geometry allowedArea1 = new WKTReader().read(AREA_WKT);
            Geometry allowedArea2 = new WKTReader().read(AREA_WKT_2);
            MultiPolygon unionedArea = (MultiPolygon)allowedArea1.union(allowedArea2);
            Request req = new Request();
            req.setService("WMS");
            req.setRequest("GetMap");
            Dispatcher.REQUEST.set(req);
            this.logout();
            this.login("anonymousUser", "", new String[]{"ROLE_ANONYMOUS", "ROLE_ANONYMOUS2"});
            unionedArea.normalize();
            VectorAccessLimits vl = (VectorAccessLimits)this.accessManager.getAccessLimits(user, lakes);
            Intersects intersects = (Intersects)vl.getReadFilter();
            MultiPolygon readFilterArea = (MultiPolygon)intersects.getExpression2().evaluate(null, MultiPolygon.class);
            readFilterArea.normalize();
            Intersects intersects2 = (Intersects)vl.getWriteFilter();
            MultiPolygon writeFilterArea = (MultiPolygon)intersects2.getExpression2().evaluate(null, MultiPolygon.class);
            writeFilterArea.normalize();
            Assert.assertTrue((boolean)unionedArea.equalsExact((Geometry)readFilterArea, 1.0E-14));
            Assert.assertTrue((boolean)unionedArea.equalsExact((Geometry)writeFilterArea, 1.0E-14));
            this.logout();
        }
        catch (Throwable throwable) {
            this.removeLayerGroup(group1);
            throw throwable;
        }
        this.removeLayerGroup(group1);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testLayerInGroupAreaRestriction() throws Exception {
        Long idRule = null;
        Long idRule2 = null;
        LayerGroupInfo group1 = null;
        try {
            Authentication user = this.getUser("anonymousUser", "", "ROLE_ANONYMOUS");
            this.login("admin", "geoserver", new String[]{"ROLE_ADMINISTRATOR"});
            Catalog catalog = this.getCatalog();
            LayerInfo lakes = catalog.getLayerByName(this.getLayerId(MockData.LAKES));
            LayerInfo fifteen = catalog.getLayerByName(this.getLayerId(MockData.FIFTEEN));
            group1 = this.createsLayerGroup("group31", LayerGroupInfo.Mode.NAMED, null, Arrays.asList(lakes, fifteen));
            idRule = this.support.addRule(GrantType.LIMIT, null, "ROLE_ANONYMOUS", "WMS", null, null, "group31", 3L);
            idRule2 = this.support.addRule(GrantType.LIMIT, null, "ROLE_ANONYMOUS", "WMS", null, "cite", "Lakes", 2L);
            this.support.addRuleLimits(idRule, CatalogMode.HIDE, AREA_WKT_INTERSECT_1, 4326);
            this.support.addRuleLimits(idRule2, CatalogMode.HIDE, AREA_WKT_INTERSECT_2, 4326);
            Geometry allowedArea1 = new WKTReader().read(AREA_WKT_INTERSECT_1);
            Geometry allowedArea2 = new WKTReader().read(AREA_WKT_INTERSECT_2);
            MultiPolygon intersectedArea = (MultiPolygon)Converters.convert((Object)allowedArea1.intersection(allowedArea2), MultiPolygon.class);
            intersectedArea.normalize();
            Request req = new Request();
            req.setService("WMS");
            req.setRequest("GetMap");
            Dispatcher.REQUEST.set(req);
            this.logout();
            this.login("anonymousUser", "", new String[]{"ROLE_ANONYMOUS"});
            VectorAccessLimits vl = (VectorAccessLimits)this.accessManager.getAccessLimits(user, lakes);
            Intersects intersects = (Intersects)vl.getReadFilter();
            MultiPolygon readFilterArea = (MultiPolygon)intersects.getExpression2().evaluate(null, MultiPolygon.class);
            readFilterArea.normalize();
            Intersects intersects2 = (Intersects)vl.getWriteFilter();
            MultiPolygon writeFilterArea = (MultiPolygon)intersects2.getExpression2().evaluate(null, MultiPolygon.class);
            writeFilterArea.normalize();
            Assert.assertTrue((boolean)intersectedArea.equalsExact((Geometry)readFilterArea, 1.0E-14));
            Assert.assertTrue((boolean)intersectedArea.equalsExact((Geometry)writeFilterArea, 1.0E-14));
            this.logout();
        }
        catch (Throwable throwable) {
            this.removeLayerGroup(group1);
            throw throwable;
        }
        this.removeLayerGroup(group1);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testLayerDeniedInGroup() throws Exception {
        Long idRule = null;
        LayerGroupInfo group1 = null;
        try {
            Authentication user = this.getUser("anonymousUser", "", "ROLE_ANONYMOUS");
            this.login("admin", "geoserver", new String[]{"ROLE_ADMINISTRATOR"});
            Catalog catalog = this.getCatalog();
            LayerInfo lakes = catalog.getLayerByName(this.getLayerId(MockData.LAKES));
            LayerInfo fifteen = catalog.getLayerByName(this.getLayerId(MockData.FIFTEEN));
            group1 = this.createsLayerGroup("group31", LayerGroupInfo.Mode.NAMED, null, Arrays.asList(lakes, fifteen));
            idRule = this.support.addRule(GrantType.LIMIT, null, "ROLE_ANONYMOUS", "WMS", null, null, "group31", 3L);
            this.support.addRule(GrantType.DENY, null, "ROLE_ANONYMOUS", "WMS", null, "cite", "Lakes", 2L);
            this.support.addRuleLimits(idRule, CatalogMode.HIDE, AREA_WKT_INTERSECT_1, 4326);
            Geometry allowedArea1 = new WKTReader().read(AREA_WKT_INTERSECT_1);
            Geometry allowedArea2 = new WKTReader().read(AREA_WKT_INTERSECT_2);
            MultiPolygon intersectedArea = (MultiPolygon)Converters.convert((Object)allowedArea1.intersection(allowedArea2), MultiPolygon.class);
            intersectedArea.normalize();
            Request req = new Request();
            req.setService("WMS");
            req.setRequest("GetMap");
            Dispatcher.REQUEST.set(req);
            this.logout();
            this.login("anonymousUser", "", new String[]{"ROLE_ANONYMOUS"});
            VectorAccessLimits vl = (VectorAccessLimits)this.accessManager.getAccessLimits(user, lakes, Arrays.asList(group1));
            Assert.assertEquals((Object)Filter.EXCLUDE, (Object)vl.getReadFilter());
            Assert.assertEquals((Object)Filter.EXCLUDE, (Object)vl.getWriteFilter());
            this.logout();
        }
        catch (Throwable throwable) {
            this.removeLayerGroup(group1);
            throw throwable;
        }
        this.removeLayerGroup(group1);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testLayerInGroupDirectAccessLimitResolutionByRole() throws Exception {
        Long idRule = null;
        Long idRule2 = null;
        LayerGroupInfo group1 = null;
        GeoFenceConfigurationManager configurationManager = (GeoFenceConfigurationManager)applicationContext.getBean("geofenceConfigurationManager", GeoFenceConfigurationManager.class);
        GeoFenceConfiguration config = configurationManager.getConfiguration();
        config.setUseRolesToFilter(true);
        config.getRoles().add("ROLE_ONE");
        try {
            Authentication user = this.getUser("aUser", "", "ROLE_ONE", "ROLE_TWO");
            this.login("admin", "geoserver", new String[]{"ROLE_ADMINISTRATOR"});
            Catalog catalog = this.getCatalog();
            LayerInfo lakes = catalog.getLayerByName(this.getLayerId(MockData.LAKES));
            LayerInfo fifteen = catalog.getLayerByName(this.getLayerId(MockData.FIFTEEN));
            group1 = this.createsLayerGroup("group31", LayerGroupInfo.Mode.NAMED, null, Arrays.asList(lakes, fifteen));
            idRule = this.support.addRule(GrantType.LIMIT, null, "ROLE_ONE", "WMS", null, null, "group31", 3L);
            idRule2 = this.support.addRule(GrantType.LIMIT, null, "ROLE_TWO", "WMS", null, "cite", "Lakes", 2L);
            this.support.addRuleLimits(idRule, CatalogMode.HIDE, AREA_WKT_INTERSECT_1, 4326);
            this.support.addRuleLimits(idRule2, CatalogMode.HIDE, AREA_WKT_INTERSECT_2, 4326);
            Geometry allowedArea1 = new WKTReader().read(AREA_WKT_INTERSECT_1);
            MultiPolygon intersectedArea = (MultiPolygon)Converters.convert((Object)allowedArea1, MultiPolygon.class);
            intersectedArea.normalize();
            this.logout();
            Request req = new Request();
            req.setService("WMS");
            req.setRequest("GetMap");
            Dispatcher.REQUEST.set(req);
            this.login("anonymousUser", "", new String[]{"ROLE_ONE", "ROLE_TWO"});
            VectorAccessLimits vl = (VectorAccessLimits)this.accessManager.getAccessLimits(user, lakes);
            Intersects intersects = (Intersects)vl.getReadFilter();
            MultiPolygon readFilterArea = (MultiPolygon)intersects.getExpression2().evaluate(null, MultiPolygon.class);
            readFilterArea.normalize();
            Assert.assertTrue((boolean)intersectedArea.equalsExact((Geometry)readFilterArea, 1.0E-14));
            this.logout();
        }
        catch (Throwable throwable) {
            this.removeLayerGroup(group1);
            config.setUseRolesToFilter(false);
            config.getRoles().remove("ROLE_ONE");
            throw throwable;
        }
        this.removeLayerGroup(group1);
        config.setUseRolesToFilter(false);
        config.getRoles().remove("ROLE_ONE");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testLayerInGroupLimitResolutionByRole() throws Exception {
        Long idRule = null;
        Long idRule2 = null;
        LayerGroupInfo group1 = null;
        GeoFenceConfigurationManager configurationManager = (GeoFenceConfigurationManager)applicationContext.getBean("geofenceConfigurationManager", GeoFenceConfigurationManager.class);
        GeoFenceConfiguration config = configurationManager.getConfiguration();
        config.setUseRolesToFilter(true);
        config.getRoles().add("ROLE_TWO");
        try {
            Authentication user = this.getUser("aUser", "", "ROLE_ONE", "ROLE_TWO");
            this.login("admin", "geoserver", new String[]{"ROLE_ADMINISTRATOR"});
            Catalog catalog = this.getCatalog();
            LayerInfo lakes = catalog.getLayerByName(this.getLayerId(MockData.LAKES));
            LayerInfo fifteen = catalog.getLayerByName(this.getLayerId(MockData.FIFTEEN));
            group1 = this.createsLayerGroup("group31", LayerGroupInfo.Mode.NAMED, null, Arrays.asList(lakes, fifteen));
            idRule = this.support.addRule(GrantType.LIMIT, null, "ROLE_ONE", "WMS", null, null, "group31", 3L);
            idRule2 = this.support.addRule(GrantType.LIMIT, null, "ROLE_TWO", "WMS", null, "cite", "Lakes", 2L);
            this.support.addRuleLimits(idRule, CatalogMode.HIDE, AREA_WKT_INTERSECT_1, 4326);
            this.support.addRuleLimits(idRule2, CatalogMode.HIDE, AREA_WKT_INTERSECT_2, 4326);
            Geometry allowedArea2 = new WKTReader().read(AREA_WKT_INTERSECT_2);
            MultiPolygon intersectedArea = (MultiPolygon)Converters.convert((Object)allowedArea2, MultiPolygon.class);
            intersectedArea.normalize();
            this.logout();
            Request req = new Request();
            req.setService("WMS");
            req.setRequest("GetMap");
            Dispatcher.REQUEST.set(req);
            this.login("anonymousUser", "", new String[]{"ROLE_ONE", "ROLE_TWO"});
            VectorAccessLimits vl = (VectorAccessLimits)this.accessManager.getAccessLimits(user, lakes, Arrays.asList(group1));
            Intersects intersects = (Intersects)vl.getReadFilter();
            MultiPolygon readFilterArea = (MultiPolygon)intersects.getExpression2().evaluate(null, MultiPolygon.class);
            readFilterArea.normalize();
            Assert.assertTrue((boolean)intersectedArea.equalsExact((Geometry)readFilterArea, 1.0E-14));
            this.logout();
        }
        catch (Throwable throwable) {
            this.removeLayerGroup(group1);
            config.setUseRolesToFilter(false);
            config.getRoles().remove("ROLE_TWO");
            throw throwable;
        }
        this.removeLayerGroup(group1);
        config.setUseRolesToFilter(false);
        config.getRoles().remove("ROLE_TWO");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testLayerInGroupAreaRestrictionRulesByUser() throws Exception {
        Long idRule = null;
        Long idRule2 = null;
        LayerGroupInfo group1 = null;
        try {
            Authentication user = this.getUser("user1", "", "ROLE1");
            this.login("admin", "geoserver", new String[]{"ROLE_ADMINISTRATOR"});
            Catalog catalog = this.getCatalog();
            LayerInfo lakes = catalog.getLayerByName(this.getLayerId(MockData.LAKES));
            LayerInfo fifteen = catalog.getLayerByName(this.getLayerId(MockData.FIFTEEN));
            group1 = this.createsLayerGroup("group71", LayerGroupInfo.Mode.NAMED, null, Arrays.asList(lakes, fifteen));
            idRule = this.support.addRule(GrantType.LIMIT, "user1", null, "WMS", null, null, "group71", 20L);
            idRule2 = this.support.addRule(GrantType.LIMIT, "user1", null, "WMS", null, "cite", "Lakes", 21L);
            this.support.addRuleLimits(idRule, CatalogMode.HIDE, AREA_WKT_INTERSECT_1, 4326);
            this.support.addRuleLimits(idRule2, CatalogMode.HIDE, AREA_WKT_INTERSECT_2, 4326);
            Geometry allowedArea1 = new WKTReader().read(AREA_WKT_INTERSECT_1);
            Geometry allowedArea2 = new WKTReader().read(AREA_WKT_INTERSECT_2);
            MultiPolygon intersectedArea = (MultiPolygon)Converters.convert((Object)allowedArea1.intersection(allowedArea2), MultiPolygon.class);
            intersectedArea.normalize();
            Request req = new Request();
            req.setService("WMS");
            req.setRequest("GetMap");
            Dispatcher.REQUEST.set(req);
            this.logout();
            this.login("user1", "", new String[]{"ROLE1"});
            VectorAccessLimits vl = (VectorAccessLimits)this.accessManager.getAccessLimits(user, lakes);
            Intersects intersects = (Intersects)vl.getReadFilter();
            MultiPolygon readFilterArea = (MultiPolygon)intersects.getExpression2().evaluate(null, MultiPolygon.class);
            readFilterArea.normalize();
            Intersects intersects2 = (Intersects)vl.getWriteFilter();
            MultiPolygon writeFilterArea = (MultiPolygon)intersects2.getExpression2().evaluate(null, MultiPolygon.class);
            writeFilterArea.normalize();
            Assert.assertTrue((boolean)intersectedArea.equalsExact((Geometry)readFilterArea, 1.0E-14));
            Assert.assertTrue((boolean)intersectedArea.equalsExact((Geometry)writeFilterArea, 1.0E-14));
            this.logout();
        }
        catch (Throwable throwable) {
            this.removeLayerGroup(group1);
            throw throwable;
        }
        this.removeLayerGroup(group1);
    }

    private void assertAdminAccess(Authentication user, WorkspaceInfo ws, boolean expectedAdmin) {
        WorkspaceAccessLimits userAccess = this.accessManager.getAccessLimits(user, ws);
        Assert.assertEquals((String)"Unexpected admin access", (Object)expectedAdmin, (Object)userAccess.isAdminable());
    }

    protected Authentication getUser(String username, String password, String ... roles) {
        ArrayList<SimpleGrantedAuthority> l = new ArrayList<SimpleGrantedAuthority>();
        for (String role : roles) {
            l.add(new SimpleGrantedAuthority(role));
        }
        return new UsernamePasswordAuthenticationToken((Object)username, (Object)password, l);
    }

    protected void setUser(Authentication user) {
        SecurityContextHolder.getContext().setAuthentication(user);
    }

    protected LayerGroupInfo createsLayerGroup(String name, LayerGroupInfo.Mode mode, LayerInfo rootLayer, List<LayerInfo> layers) throws Exception {
        return this.createsLayerGroup(name, mode, rootLayer, layers, null);
    }

    protected LayerGroupInfo createsLayerGroup(String name, LayerGroupInfo.Mode mode, LayerInfo rootLayer, List<LayerInfo> layers, CoordinateReferenceSystem groupCRS) throws Exception {
        Catalog catalog = this.getCatalog();
        LayerGroupInfo group = catalog.getFactory().createLayerGroup();
        group.setName(name);
        group.setMode(mode);
        if (rootLayer != null) {
            group.setRootLayer(rootLayer);
            group.setRootLayerStyle(rootLayer.getDefaultStyle());
        }
        for (LayerInfo li : layers) {
            group.getLayers().add(li);
        }
        group.getStyles().add(null);
        group.getStyles().add(null);
        CatalogBuilder cb = new CatalogBuilder(catalog);
        cb.calculateLayerGroupBounds(group);
        if (groupCRS != null) {
            ReferencedEnvelope re = group.getBounds();
            MathTransform transform = CRS.findMathTransform((CoordinateReferenceSystem)group.getBounds().getCoordinateReferenceSystem(), (CoordinateReferenceSystem)groupCRS);
            Envelope bbox = JTS.transform((Envelope)re, (MathTransform)transform);
            ReferencedEnvelope newRe = new ReferencedEnvelope(bbox.getMinX(), bbox.getMaxX(), bbox.getMinY(), bbox.getMaxY(), groupCRS);
            group.setBounds(newRe);
        }
        catalog.add(group);
        return group;
    }

    private void removeLayerGroup(LayerGroupInfo ... groups) {
        this.login("admin", "geoserver", new String[]{"ROLE_ADMINISTRATOR"});
        for (LayerGroupInfo group : groups) {
            if (group == null) continue;
            this.getCatalog().remove(group);
        }
        this.logout();
    }

    private String repWkt(String srcWKT, CoordinateReferenceSystem srcCRS, CoordinateReferenceSystem targetCRS, int targetSRID) throws ParseException, FactoryException, TransformException {
        Geometry geometry = new WKTReader().read(srcWKT);
        MathTransform mt = CRS.findMathTransform((CoordinateReferenceSystem)srcCRS, (CoordinateReferenceSystem)targetCRS, (boolean)true);
        Geometry transformed = JTS.transform((Geometry)geometry, (MathTransform)mt);
        transformed.setSRID(targetSRID);
        return new WKTWriter().write(transformed);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testLayerBothAreas() throws Exception {
        LayerGroupInfo group1 = null;
        try {
            Authentication user = this.getUser("anonymousUser", "", "ROLE_ANONYMOUS", "ROLE_ANONYMOUS2");
            this.login("admin", "geoserver", new String[]{"ROLE_ADMINISTRATOR"});
            Catalog catalog = this.getCatalog();
            LayerInfo lakes = catalog.getLayerByName(this.getLayerId(MockData.LAKES));
            LayerInfo fifteen = catalog.getLayerByName(this.getLayerId(MockData.FIFTEEN));
            group1 = this.createsLayerGroup("groupTree31", LayerGroupInfo.Mode.NAMED, null, Arrays.asList(lakes, fifteen));
            Long idRule = this.support.addRule(GrantType.LIMIT, null, "ROLE_ANONYMOUS", "WMS", null, "cite", "Lakes", 3L);
            Long idRule2 = this.support.addRule(GrantType.LIMIT, null, "ROLE_ANONYMOUS2", "WMS", null, "cite", "Lakes", 2L);
            this.support.addRuleLimits(idRule, CatalogMode.HIDE, AREA_WKT, 4326, SpatialFilterType.CLIP);
            this.support.addRuleLimits(idRule2, CatalogMode.HIDE, AREA_WKT_2, 4326, SpatialFilterType.INTERSECT);
            Request req = new Request();
            req.setService("WMS");
            req.setRequest("GetMap");
            Dispatcher.REQUEST.set(req);
            this.logout();
            this.login("anonymousUser", "", new String[]{"ROLE_ANONYMOUS", "ROLE_ANONYMOUS2"});
            VectorAccessLimits vl = (VectorAccessLimits)this.accessManager.getAccessLimits(user, lakes);
            Assert.assertNotNull((Object)vl.getClipVectorFilter());
            Assert.assertNotNull((Object)vl.getIntersectVectorFilter());
            this.logout();
        }
        catch (Throwable throwable) {
            this.removeLayerGroup(group1);
            throw throwable;
        }
        this.removeLayerGroup(group1);
    }
}

