/* JAI-Ext - OpenSource Java Advanced Image Extensions Library
 *    http://www.geo-solutions.it/
 *    Copyright 2018 GeoSolutions
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 *  Copyright (c) 2011, Michael Bedward. All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without modification,
 *  are permitted provided that the following conditions are met:
 *
 *  - Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 *
 *  - Redistributions in binary form must reproduce the above copyright notice, this
 *    list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 *  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 *  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 *  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package it.geosolutions.jaiext.jiffle.runtime;

import java.awt.*;
import java.awt.image.RenderedImage;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import it.geosolutions.jaiext.jiffle.JiffleException;

/**
 * The default abstract base class for runtime classes that implement indirect evaluation.
 *
 * @author Michael Bedward
 * @since 0.1
 * @version $Id$
 */
public abstract class AbstractIndirectRuntime extends AbstractJiffleRuntime
        implements JiffleIndirectRuntime {

    /*
     * Note: not using generics here because they are not
     * supported by the Janino compiler.
     */
    List sourceImageNames = new ArrayList();
    String destImageName;
    Integer destinationBands;

    /** Creates a new instance and initializes script-option variables. */
    public AbstractIndirectRuntime(String[] variableNames) {
        super(variableNames);
        initOptionVars();
    }

    public void setDestinationImage(String varName) {
        try {
            doSetDestinationImage(varName, null);
        } catch (WorldNotSetException ex) {
            // Passing a null transform does not cause an Exception
        }
    }

    public void setDestinationImage(String varName, CoordinateTransform tr) throws JiffleException {
        try {
            doSetDestinationImage(varName, tr);
        } catch (WorldNotSetException ex) {
            throw new JiffleException(
                    String.format(
                            "Setting a coordinate tranform for a source (%s) without"
                                    + "having first set the world bounds and resolution",
                            varName));
        }
    }

    private void doSetDestinationImage(String varName, CoordinateTransform tr)
            throws WorldNotSetException {

        destImageName = varName;
    }

    /** {@inheritDoc} */
    public void setDefaultBounds() {
        String imageName = _images.keySet().iterator().next();
        RenderedImage refImage = _images.get(imageName).image;

        Rectangle rect =
                new Rectangle(
                        refImage.getMinX(), refImage.getMinY(),
                        refImage.getWidth(), refImage.getHeight());

        setWorldByResolution(rect, 1, 1);
    }

    @Override
    public void setDestinationBands(Map<String, Integer> destinationBands) {
        if (destinationBands.size() > 1 || destinationBands.isEmpty()) {
            throw new IllegalStateException(
                    "The indirect runtime works with one destination image only");
        }
        this.destinationBands = destinationBands.values().iterator().next();
    }

    @Override
    public int getOutputBands() {
        return destinationBands;
    }
}
